//global.h: includes all objects related to the simulator itself
#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "resources/menu_ids.h"

class disaster;//Forward-declare this, so we can use it later
class person;//Forward-declare this, too, so we can use it later
class gridCell;

//********************* Utility Classes/ Typedefs *********************
typedef double coord;//A coordinate in simulator space
typedef double real;//A real number (might use "float" for this)

//Point: a 2D coordinate pair.  Can also be thought of as a vector
class point {
public:
	coord x,y;//Location of the point
	point() {x=y=0;}
	point(coord Nx,coord Ny)	{x=Nx;y=Ny;}
	//(Default copy constructor)
	void add(point &p)	{x+=p.x;y+=p.y;}
	void sub(point &p)	{x-=p.x;y-=p.y;}
	void scale(real r)	{x*=r;y*=r;}
	void div(real r)	{scale(1.0/r);}
	real dot(point &p)	{return p.x*x+p.y*y;}
	real mag(void) {return sqrt(dot(*this));}
	void normalize(void) {scale(1.0/mag());}
	friend real ptDist(point &a,point &b)
		{point c=a;c.sub(b);return c.mag();}
	friend real moreDist(point &a,point &b)//Is at least ptDist
		{real dx=a.x-b.x,dy=a.y-b.y;
		if (dx<0)dx=-dx;if (dy<0)dy=-dy;return dx+dy;}
	friend point blend(point a,real strengthA,point b)
	{point ret=a,bScale=b;ret.scale(strengthA);
	bScale.scale(1.0-strengthA);ret.add(bScale);return ret;}
};
//Point utility routines
bool intersects(point a1,point a2,point b1,point b2,point &intersection);
coord lineDist(person &p,point t1,point t2,coord width,point &hitPt);
//Return the distance between the given person and the line t1-t2,
// with given line width.

//Win32 Pens used for coloring various objects
extern HPEN blackPen,redPen,bluePen,greenPen;

//Random number utilities
real realRand(void);//Return a random number on [0,1)
point pointRand(real max);//Return random point on 
                    //[-max/2,max/2)x[-max/2,max/2)
bool probability(int percent);//Return true every "probability" out of 100 tries


//Screen: Map from simulator "coord"s to screen locations; draw to screen
class screen {
public:
	HDC dc;
	double scaleX,offX,scaleY,offY;
	screen(HDC Ndc,
			double NscaleX,double NoffX,
			double NscaleY,double NoffY)
		{dc=Ndc;
		scaleX=NscaleX;offX=NoffX;
		scaleY=NscaleY;offY=NoffY;}
	void printFloat(point p,double f);
	void printStr(point p,const char *str);
	void map(point p,int &x,int &y) {x=(int)(p.x*scaleX+offX);y=(int)(p.y*scaleY+offY);}
};


//**************************************************************************************
//**************** Simulation Classes ****************************

//Drawable: Abstract Superclass of all objects that can be drawn to a window
class drawable {
public:
		virtual void draw(screen &s)=0;
		virtual HPEN pen(void) {return blackPen;}
};

//Hit: These are things people can trip over/run into (including other people)
class hit: public drawable {
public:
	virtual coord hits(person &p,point &hitPt)=0;
	//Return nearest hit point, and the distance 
	//between person's edge and this object.
	//(if negative, person is penetrating object)
	virtual bool inWay(person *you,point dest,coord &dist) {return false;}
	//Are we in the way between you and dest?
	//If so, how far away are we from you?
	virtual point reRoute(person *you,point dest) {return dest;}
	//Give me a better route between you and dest,
	//so I don't hit you.
};

//Wall: One straight line segment, comprising a barrier to travel and sight.
class wall: public hit {
public:
	point t1,t2;//Endpoints of wall
	real radius;//Size of "Repulsion Field" around wall
	virtual void init(point Nt1,point Nt2) {t1=Nt1;t2=Nt2;radius=0.0;}
	virtual void draw(screen &s);
	virtual coord hits(person &p,point &hitPt);//Return nearest hit point, and distance to there
};

//Anexit: An exit is a wall that eats people
class anexit: public wall {
public:
	point myCenter;
	int nFolks;//Number of people surrounding exit
	double congestion;
	double myStrength;
	coord diameter;//Search diameter for congestion
	virtual bool iamBetter(anexit *other,point vantage);//Is this exit better than other?
	virtual void calcCongestion(disaster &d);//Find out how many people are nearby
	virtual double strength(point vantage);
	virtual void init(point Nt1,point Nt2,double Nstrength,double Ndiameter);
	virtual void draw(screen &s);
	virtual HPEN pen(void) {return greenPen;}
	bool canLeave(person &p);//Can person p leave via this exit now?
	point center(void) {return myCenter;}//Return the coordinates of the exit's center
};

//SquareFurniture: A piece of square furniture, like a desk
class squareFurniture: public hit {
public:
	point v[5];//Vertices (v[0]==v[4])
	point c;//Center
	virtual void init(coord len,coord ht,coord x,coord y,coord theta);
	virtual void draw(screen &s);
	virtual coord hits(person &p,point &hitPt);
	virtual bool inWay(person *you,point dest,coord &dist);
	virtual point reRoute(person *you,point dest);
};

//RoundFurniture: A piece of round furniture
class roundFurniture: public hit {
public:
	point c;//Center of furniture
	coord radius;
	virtual void init(point Nc,coord Nradius) {c=Nc;radius=Nradius;}
	virtual void draw(screen &s);
	virtual coord hits(person &p,point &hitPt);
	virtual bool inWay(person *you,point dest,coord &dist);
	virtual point reRoute(person *you,point dest);
};

//Person: A person is basically a piece of round furniture that moves
//This way, the hit detection code only need be written once.
class person: public roundFurniture {
public:
	typedef enum {
		state_walking=0,//Walking toward exit
		state_noexit=1,//Can't see an exit
		state_atexit=2//Am sitting in exit
	} stateType;
	gridCell *cell;
	coord totalProgress,lastProgress;
	anexit *destExit;
	int nStuck;point lastStuck;
	point lastDest;
	point walkDir;
	person() {destExit=NULL;cell=NULL;totalProgress=lastProgress=0;}
	virtual coord walkSpeed(disaster &d);//Return walking speed during given disaster
	point getDestination(disaster &d);//Find nearest exit from given disaster
	bool repelledAt(hit *h,point newPos,point &repulsion);
	bool hitTest(hit *h);//Do we run into h walking in this direction?
	virtual HPEN pen(void);

	void advance_begin(disaster &d);//Flee the given disaster
	bool hitPeople(person *p);//Do we run into p walking in this direction?
	void advance_end(disaster &d);//Done fleeing
	bool at_exit(disaster &d);//Out of the given exit
	virtual void draw(screen &s);//Draw self on screen
};

class gridBag;//A set of grid cells, forward-declared for efficiency

//Disaster: contains the state of the evacuation of a single room.
#define max_objects 100 //The 
#define max_people 5000 //The maximum # of occupants (for static alloc)
class disaster {
public:
	real dt;/*Time step, in seconds*/
	int time;/*Total number of frames so far*/
	gridBag *grid;
	//Lists of objects in the room
	wall **walls;int nwalls;
	anexit **exits;int nexits;
	hit **hits;int nhits;
	person **people;int npeople;
	
	void init(const char *fName);
	~disaster();
	bool isVisible(point dest,point source);//Is dest visible from source?
	bool parse_input(const char *fName);//Add objects in given input file to disaster
	void advance(void);//Move objects around by one simulation timestep
	void draw(screen &s);//Draw all objects (may be much slower than "advance()")
	int totalSaved;//Total number of people that have exited room
	FILE *log,*log2;//Log file pointers
	void saveLog(void);//Write important data to log file
};

#include "app_main.h"

#include "grid.h"
