/*
Very simple GLUT program using OSL libraries.

Orion Sky Lawlor, olawlor@acm.org, 2005/1/26 (Public Domain)
*/
#include "osl/viewpoint.h" /* 3D viewpoint utility routines */
#include "osl/viewpoint.cpp" /* include implementation code, for easier linking */
#include <GL/glut.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>

/*  This is an array of keyboard flags, indexed by key.
  Initially 0; corresponding keyboard press toggles to 1.
  For example, the x key controls oglToggles['x'].  */
int oglToggles[256]; 

/*********** OpenGL Camera Definition ********/
osl::Axes3d cameraDir; /* Camera axes: Z points to camera, X and Y on screen */
double cameraDist=5.0; /* Distance to camera from cameraTarget */
osl::Vector3d cameraTarget(0,0,0); /* point camera is looking at, world coords */

/* Call this whenever any camera parameters change.
   Updates the MODELVIEW matrix with the current camera location. */
void cameraChanged(void) {
	osl::Vector3d cameraLocation=cameraTarget+cameraDist*cameraDir.getZ(); /* location of the camera, world coords */
	glLoadIdentity(); /* because gluLookAt adds to current matrix */
	gluLookAt(cameraLocation.x,cameraLocation.y,cameraLocation.z,
		cameraTarget.x,cameraTarget.y,cameraTarget.z,
		cameraDir.getY().x,cameraDir.getY().y,cameraDir.getY().z);
	
	/* After changing the modelview matrix, you've got to set up the 
	OpenGL light positions again, because light positions are *specified*
	in model coordinates, but OpenGL *keeps* them in eye coordinates.  
	Might as well just set up all lighting here:
	*/
	
/* Set up OpenGL lighting (*lots* of stuff to set up) */
	glEnable(GL_LIGHTING);
	glEnable(GL_LIGHT0);
	osl::Vector3d l(0.1,1.0,-0.3); l=l.dir();
	float lightDir[4];
	lightDir[0]=l.x; lightDir[1]=l.y; lightDir[2]=l.z; lightDir[3]=0.0;
	glLightfv(GL_LIGHT0, GL_POSITION, lightDir);

	GLfloat ones[] = {1.0f, 1.0f, 1.0f, 0.0f};
	GLfloat specularity[] = {50.0f};
	if (oglToggles['s']) specularity[0]=100.0;
	glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT,   ones);
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,   ones);
	glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR,  ones);
	glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS,  specularity);

	glEnable(GL_COLOR_MATERIAL);
	glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
	// glLightModel(GL_LIGHT_MODEL_COLOR_CONTROL,GL_SEPARATE_SPECULAR_COLOR);
	glEnable(GL_NORMALIZE);
	
	
	glutPostRedisplay(); /* ask GLUT to redraw our display */
}

void reshapeFn(int w, int h) /* Called by GLUT when window is resized */
{
	/* GL_PROJECTION matrix contains only the camera parameters */
	glViewport(0, 0, (GLsizei)w, (GLsizei)h);
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity(); /* because gluPerspective combines with the current matrix */
	double z_near=1.0e-3,z_far=1.0e5; /* z buffer range */
	gluPerspective(70.0, w/(double)h, z_near,z_far);
	
	/* GL_MODELVIEW contains the world->camera transform */
	glMatrixMode(GL_MODELVIEW);
	cameraChanged();
}

/************ Crappy Keyboard User Interface **************/
void keyboardFn(unsigned char key, int x, int y) /* Called by GLUT when a key is pressed */
{
	switch (key) {
	case 27: case 'x': case 'q': /*  Escape or quit key  */
		exit(0);
		break;
	case '+': case '=': cameraDist*=0.94; break; /* zoom in */
	case '-': case '_': cameraDist*=1.05; break; /* zoom out */
	default: /* all other keys just toggle the corresponding "keys" entry */
		oglToggles[key]=!oglToggles[key]; 
		break;
	}
	cameraChanged(); /* ask GLUT to redraw our display */
}

/************ Crappy Mouse User Interface **************/

int lastX,lastY;
enum {mouseNone=0,mouseSpin,mouseZoom} mouseState;

void motionFn(int x, int y)
{
	double s=0.005; /* spin rate: radians of rotation per pixel of mouse movement */
	double z=0.01; /* zoom rate: zoom scale factor per pixel */
	int relX=x-lastX, relY=y-lastY;
	switch (mouseState) {
	case mouseSpin: cameraDir.nudge(relX*s,-relY*s); cameraChanged(); break;
	case mouseZoom: cameraDist*=1.0+z*relY; cameraChanged(); break;
	case mouseNone: break; /* no buttons down-- don't do anything */
	}
	lastX=x; lastY=y;
}

void mouseFn(int button, int state, int x, int y)
{
	lastX=x; lastY=y;
	if (state==GLUT_DOWN) {mouseState=(button==0)?mouseSpin:mouseZoom;}
	if (state==GLUT_UP) {mouseState=mouseNone;}
}

/************ Main Drawing Function **************/
void displayFn(void) /* Called by GLUT to show every frame */
{
/* Do basic OpenGL initialization */
	glEnable(GL_DEPTH_TEST);
	glEnable(GL_BLEND); /* alpha blending */
	glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glShadeModel (GL_SMOOTH); /* gauroud (not flat) shading */
	glClearColor (0.0, 0.0, 0.0, 0.0); /* sets background color (RGBA) */
	
/* Do any funny overrides */
	if (oglToggles['z']) glDisable(GL_DEPTH_TEST); /* 'z' disables the Z-Buffer */
	if (oglToggles['f']) glShadeModel(GL_FLAT); /* 'f' changes to flat (not smooth) shading */
	if (oglToggles['b']) glClearColor(1,0,0,0); /* 'b' changes the background color */

/* Main redraw */
	if (!oglToggles['e']) /* 'e' key turns off erase before screen draw */
		glClear(GL_COLOR_BUFFER_BIT+GL_DEPTH_BUFFER_BIT); /* clears to background color */
	
	glPushMatrix(); /* keeps the "translate" below from accumulating frame-to-frame */

	glColor3f(1,0,0); /* big red sphere */
	glutSolidSphere(1.0,40,60);
	
	glTranslatef(1.5,0,0); /* little green sphere */
	glColor3f(0,1,0);
	glutSolidSphere(0.3,40,60);

	glPopMatrix();
	if (!oglToggles['s']) /* 's' turns off swapbuffers */
		glutSwapBuffers(); /* displays rendered image */
}


/******** Program Initialization *********/
int main(int argc, char** argv)
{
/* Do GLUT initialization */
	glutInit(&argc, argv);
	glutInitDisplayMode (GLUT_DOUBLE | GLUT_DEPTH | GLUT_RGBA);
	glutInitWindowSize (1000, 700);
	glutCreateWindow (argv[0]); /* could also do glutEnterGameMode for fullscreen */

/* Set up GLUT GUI routines */
	glutDisplayFunc (displayFn);
	glutReshapeFunc (reshapeFn);
	glutKeyboardFunc (keyboardFn);
	glutMouseFunc(mouseFn);
	glutMotionFunc(motionFn);
	
	/* Run the user interface until exit.  This call never returns. */
	glutMainLoop();
	return 0;
}
