/*
GL_ARB_fragment_program (or GL_ARB_vertex_program) per-pixel
or per-vertex program bindings.

Orion Sky Lawlor, olawlor@acm.org, 8/13/2002 (Public Domain)
*/
#ifndef __OGL_PROGRAM_H
#define __OGL_PROGRAM_H

#include "ogl/ext.h" /* MUST call oglSetupExt if you use this header! */
#include "ogl/util.h"

/**
  Program to run on each vertex or pixel fragment.
*/
class oglProgram {
	int target;// Program target, like GL_FRAGMENT_PROGRAM_ARB
	GLuint progName; // OpenGL handle to program
	void make(int target_);
public:
	oglProgram(int target_=GL_FRAGMENT_PROGRAM_ARB) {make(target_);}
	oglProgram(const char *str,int target_=GL_FRAGMENT_PROGRAM_ARB)
		{make(target_); set(str);}
	
	/// Make this the current OpenGL program.
	void bind(void) {
		glBindProgramARB(target, progName);
	}
	/// Enable this program-- bind it and enable rendering
	void begin(void) {
		bind();
		glEnable(target);
	}
	/// Stop rendering using this program.
	void end(void) {
		glDisable(target);
	}
	
	// Replace the current program by the one in this file.
	//  Prints out an error message if the program can't be parsed.
	void read(const char *fName);
	
	// Replace the current program by this one.
	//  Prints out an error message if the program can't be parsed.
	void set(const char *programText);
	
	~oglProgram() {
		glDeleteProgramsARB(1,&progName);
	}
};

/**
 Defines a vertex program that computes the points' 
 world coordinates, normal, albedo, and viewing direction.
 You must finish this with "END", like
 oglProgram vp(GL_ARB_VERTEX_PROGRAM); vp.set(oglProgram_vertexStandard "END");
*/
#define oglProgram_vertexStandard \
"!!ARBvp1.0\n"\
"#\n"\
"# Demonstrates a simple vertex shader to compute world coordinates.\n"\
"#\n"\
"# Orion Sky Lawlor, olawlor@acm.org, 2005/2/7 (Public Domain)\n"\
"#\n"\
"\n"\
"# Program/vertex communication protocol\n"\
"PARAM  camera =program.env[0]; # World-coordinates eye location\n"\
"\n"\
"# Vertex/fragment communication protocol\n"\
"OUTPUT world  =result.texcoord[4]; # World-coordinates position\n"\
"OUTPUT worldN =result.texcoord[5]; # World-coordinates normal\n"\
"OUTPUT albedo =result.texcoord[6]; # Material \"diffuse color\"\n"\
"OUTPUT viewdir=result.texcoord[7]; # Viewing direction (points toward eye)\n"\
"\n"\
"# Compute clip coordinates by the multiplying by modelview & projection matrix.\n"\
"#  To draw geometry, OpenGL requires vertex programs to do this. \n"\
"PARAM  mvp[4]	    = { state.matrix.mvp };\n"\
"# Map vertex to clip coordinates, by multiplying by mvp matrix:\n"\
"\n"\
"TEMP P;\n"\
"DP4   P.x, mvp[0], vertex.position; \n"\
"DP4   P.y, mvp[1], vertex.position;\n"\
"DP4   P.z, mvp[2], vertex.position;\n"\
"DP4   P.w, mvp[3], vertex.position;\n"\
"# MUL P.x,P.x,0.3;\n"\
"MOV result.position, P;\n"\
"\n"\
"\n"\
"# Compute world position of vertex by multiplying by modelview\n"\
"PARAM  mv[4]	    = { state.matrix.modelview };\n"\
"TEMP W;\n"\
"DP4   W.x, mv[0], vertex.position; \n"\
"DP4   W.y, mv[1], vertex.position;\n"\
"DP4   W.z, mv[2], vertex.position;\n"\
"MOV   W.w, 1;\n"\
"MOV world,W;\n"\
"\n"\
"# Compute world normal of vertex by multiplying by modelview inverse transpose\n"\
"PARAM  mvi[4]	    = { state.matrix.modelview.invtrans};\n"\
"TEMP N;\n"\
"DP4   N.x, mvi[0], vertex.normal; \n"\
"DP4   N.y, mvi[1], vertex.normal;\n"\
"DP4   N.z, mvi[2], vertex.normal;\n"\
"MOV   N.w, 0;\n"\
"TEMP L;\n"\
"DP4   L.w,N,N; # Normalize N into worldN\n"\
"RSQ   L,L.w;\n"\
"MUL   worldN,N,L;\n"\
"\n"\
"# Copy over diffuse color\n"\
"MOV  albedo, vertex.color;\n"\
"\n"\
"# Compute view direction=(camera-world).normalize();\n"\
"TEMP view;\n"\
"SUB view,camera,W;\n"\
"MOV view.w,0;\n"\
"DP4   L.w,view,view; # Normalize view into viewdir\n"\
"RSQ   L,L.w;\n"\
"MUL   viewdir,view,L;\n"\
"\n"


/**
 Defines a fragment program that has access to the world coordinates,
 normal, color, and viewing direction computed by the vertex program above.
 You must at least write to result.color and add "END".
*/
#define oglProgram_fragmentStandard \
"!!ARBfp1.0\n"\
"# Demonstrates basic vertex/fragment shader communication.\n"\
"# Orion Sky Lawlor, olawlor@acm.org, 2005/2/7 (Public Domain)\n"\
"#\n"\
"\n"\
"# Vertex/fragment communication protocol\n"\
"ATTRIB world  =fragment.texcoord[4]; # World-coordinates position\n"\
"ATTRIB worldN =fragment.texcoord[5]; # World-coordinates normal\n"\
"ATTRIB albedo =fragment.texcoord[6]; # Material \"diffuse color\"\n"\
"ATTRIB viewdir=fragment.texcoord[7]; # Viewing direction (points toward eye)\n"\
"\n"




#endif /*def(thisHeader)*/

