/*
Broilerplate GLUT/OpenGL main routine.

Orion Sky Lawlor, olawlor@acm.org, 8/13/2002
*/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "ogl/main.h"
#include "ogl/readback.h"
#include "osl/vector4d.h" /* for lighting */

toggle_t oglToggles;
int oglMouseX,oglMouseY,oglMouseButtons;

oglDrawable::~oglDrawable() {}

// Default implementation of all oglEventConsumer events: just ignore it
oglEventConsumer::~oglEventConsumer() {} 
void oglEventConsumer::reshape(int wid,int ht) {}
void oglEventConsumer::mouseHover(int x,int y) {}
void oglEventConsumer::mouseDown(int button,int x,int y) {}
void oglEventConsumer::mouseDrag(int buttonMask,int x,int y) {}
void oglEventConsumer::mouseUp(int button,int x,int y) {}
void oglEventConsumer::handleEvent(int uiCode) {
	if (uiCode>0 && uiCode<256) 
	{ // flip toggles on keypress
		oglToggles[uiCode]=!oglToggles[uiCode];
		oglRepaint(10);
	}
}


// oglTrackballController:
oglTrackballController::oglTrackballController(
			double eyeDist_,double fov_,oglVector3d rotcen_)
	:rotcen(rotcen_), eyeDist(eyeDist_), fov(fov_)
{
	state=stateNone;
}
const oglViewpoint &oglTrackballController::getViewpoint(int width,int height)
{
	viewpoint=oglViewpoint(getEye(),rotcen,axes.getY());
	viewpoint.discretize(width,height,fov);

	return viewpoint;
}

void oglTrackballController::mouseDown(int button,int x,int y) {
	lastX=x; lastY=y;
	if (button==0) { /* left mouse: rotate or twirl */
		state=stateRotate;
		oglVector3d screen(viewpoint.getWidth(),viewpoint.getHeight(),0);
		double rotateR=screen.mag()*0.5*0.7;
		double r=(oglVector3d(x,y,0)-0.5*screen).mag();
		if (r>rotateR) /* outer click: twirl */
			state=stateTwirl;
	}
	else /* button != 0 */ { /* zoom */
		state=stateZoom;
	}
}
void oglTrackballController::mouseDrag(int buttonMask,int x,int y) {
	double scale=0.002;
	double dx=scale*(x-lastX);
	double dy=scale*(y-lastY);
	switch (state) {
	case stateRotate:
		axes.nudge(dx,dy);
		break;
	case stateZoom:
		eyeDist*=(1.0+dy);
		break;
	case stateTwirl: 
		{
		oglVector3d cen(0.5*viewpoint.getWidth(), 0.5*viewpoint.getHeight(), 0);
		oglVector3d rel(oglVector3d(x,y,0)-cen);
		oglVector3d tan(-rel.y,rel.x,0);
		double rotAng=-oglVector3d(x-lastX,y-lastY,0).dot(tan.dir())/tan.mag();
		axes.rotate(rotAng);
		} break;
	case stateNone:  /* for whining compilers */
		break;
	};
	oglRepaint(10);
	lastX=x; lastY=y;
}
void oglTrackballController::handleEvent(int uiCode) {
	switch(uiCode) {
	case '+': case '=': 
		eyeDist*=0.9; oglRepaint(10); break;
	case '-': case '_':
		eyeDist*=1.1; oglRepaint(10); break;
	default:
		oglEventConsumer::handleEvent(uiCode);
	}
}


/******************* GLUT interface *******************/

static oglOptions myCfg;
static oglController *myCtl=NULL;
static oglDrawable *myScene=NULL;

// oglController:
void oglController::getZRange(double &z_near,double &z_far) {
	z_near=1.0/myCfg.depthRange; z_far=myCfg.depthRange;
}
void oglTrackballController::getZRange(double &z_near,double &z_far) {
	z_near=eyeDist/myCfg.depthRange; z_far=eyeDist*myCfg.depthRange;
}

static double avgTime=0; // Recent time per frame (s)

#define MYCALLBACK(ret) extern "C" ret

MYCALLBACK(void) motionFn(int x, int y)
{
	y=myCfg.height-y; //Flip y around so it matches OpenGL coords.
	oglMouseX=x; oglMouseY=y;
	if (oglMouseButtons==0) myCtl->mouseHover(x,y);
	else /* oglMouseButtons!=0 */ myCtl->mouseDrag(oglMouseButtons,x,y);
}

MYCALLBACK(void) mouseFn(int button, int state, int x, int y)
{
	y=myCfg.height-y; //Flip y around so it matches OpenGL coords.
	oglMouseX=x; oglMouseY=y;
	if (state==GLUT_DOWN) {
		oglMouseButtons|=(1<<button); /* Set the bit for this button */
		myCtl->mouseDown(button,x,y);
	}
	if (state==GLUT_UP) {
		myCtl->mouseDrag(oglMouseButtons,x,y);
		oglMouseButtons&=~(1<<button); /* Clear the bit for this button */
		myCtl->mouseUp(button,x,y);
	}
}

/** Default state for a new frame */
void oglSetFrameState(void) 
{

	/* lights and materials */
	oglDirectionLight(GL_LIGHT0, oglVector3d(0.0, 0.0, 1.0), 
		oglVector3d(1.0, 1.0, 0.7), 0.1,0.8,0.0);
	oglDirectionLight(GL_LIGHT1, oglVector3d(0.0, 0.0, -1.0), 
		oglVector3d(0.05, 0.1, 0.2), 0.4,0.3,0.0);

	GLfloat ones[] = {1.0f, 1.0f, 1.0f, 0.0f};
	GLfloat specularity[] = {50.0f};
	glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT,   ones);
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,   ones);
	glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR,  ones);
	glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS,  specularity);

	glEnable(GL_COLOR_MATERIAL);
	glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
	/* glLightModel(GL_LIGHT_MODEL_COLOR_CONTROL,GL_SEPARATE_SPECULAR_COLOR); */
	oglEnable(myCfg.lighting, GL_LIGHTING);
	if (myCfg.lighting) {
		glEnable(GL_NORMALIZE);
	}

	/* Depth state */
	glDepthFunc(GL_LEQUAL);
	oglEnable(myCfg.depth, GL_DEPTH_TEST);
	
	glShadeModel (GL_SMOOTH);
	
	/* Alpha blending */
	glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	if (myCfg.alpha) {
		glEnable(GL_BLEND);
		glEnable(GL_ALPHA_TEST);
		glAlphaFunc(GL_GREATER,0.01f);
	}
	
	/* Textures */
	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
	// oglWrapType(GL_REPEAT); //? 
	// oglTextureType(oglTexture_mipmap); // Screws up currently bound texture-- bad idea...
}

/**
 Perform all OpenGL-related start-frame work:
   -Clear the back buffer.
   -Initialize lighting and matrices.
*/
void oglStartFrame(const oglViewpoint &vp) {
/* Clear buffers */
	int clearMode=0;
	if (myCfg.clear) clearMode|=GL_COLOR_BUFFER_BIT;
	if (myCfg.depth) clearMode|=GL_DEPTH_BUFFER_BIT;
	if (myCfg.stencil) clearMode|=GL_STENCIL_BUFFER_BIT;
	glClearColor(myCfg.clearColor.x,myCfg.clearColor.y,myCfg.clearColor.z,0);
	if (clearMode!=0) glClear(clearMode);

/* Set up the projection and model matrices */
	oglSetViewpoint(vp);
	glLoadIdentity();

/* Set up OpenGL state */
	oglSetFrameState();
	
	oglCheck("after frame setup");
}

void oglSetViewpoint(const oglViewpoint &vp) {
	glViewport(0, 0, (GLsizei) vp.getWidth(), (GLsizei) vp.getHeight());
	glMatrixMode(GL_PROJECTION);
	double mat[16],z_near=1.0e-3,z_far=1.0e5;
	if (myCtl) myCtl->getZRange(z_near,z_far);
	vp.makeOpenGL(mat,z_near,z_far);
	glLoadMatrixd(mat);
	
	glMatrixMode(GL_MODELVIEW);
}

/**
 Perform all OpenGL-related end-frame work:
   -Swap the buffers
*/
void oglEndFrame(void) {
	
	
	glutSwapBuffers();
}

/* Object to read back depth and stencil buffers */
static oglReadback rb;

MYCALLBACK(void) display(void)
{
	glFinish();
	double startFrame=oglTimer();
	const oglViewpoint &vp=myCtl->getViewpoint(myCfg.width,myCfg.height);
	oglStartFrame(vp);
	myScene->draw(vp);
	oglCheck("after scene display");

	glFinish();
	double newTime=oglTimer()-startFrame;
	double compareTime=(newTime>avgTime)?newTime:avgTime;
	/* Set it up so we average the times of previous frames with
	  a decay half-life of about 10ms: */
	double decayTime=0.01; /* seconds */
	double oldWeight;
	double maxWeight=0.99;
	if (compareTime<=0)
		oldWeight=maxWeight;
	else {
		oldWeight=0.5*(decayTime/compareTime); /* t==decayTime -> 0.5 weight */
		if (!(oldWeight>0)) oldWeight=0;
		if (!(oldWeight<maxWeight)) oldWeight=maxWeight;
	}
	// printf (" old= %f   new= %f   wgt= %f\n",avgTime,newTime,oldWeight);
	avgTime=oldWeight*avgTime + (1.0-oldWeight)*newTime;
	
	if (myCfg.framerate) {
		glDisable(GL_TEXTURE_2D);
		glDisable(GL_DEPTH_TEST);
		glLoadIdentity();
		oglPixelMatrixSentry pixelMode;
		glColor3f(1.0f,1.0f,1.0f);
		oglPrintf(oglVector3d(-0.98,-0.98,0),"%.2f ms/frame",1.0e3*avgTime);
	}
	
	if (myCfg.depth && oglToggles['D']) { /* Show depth buffer */
		rb.readDepth(0,0,vp.getWidth(),vp.getHeight());
		rb.show();
	}
	if (myCfg.stencil && oglToggles['S']) { /* Show stencil buffer */
		rb.readStencil(0,0,vp.getWidth(),vp.getHeight());
		rb.show();
	}
	
	oglEndFrame();
}

MYCALLBACK(void) reshape(int w, int h)
{
	myCfg.width=w; myCfg.height=h;
	
	myCtl->reshape(w,h);
}

/* ARGSUSED1 */
MYCALLBACK(void) keyboard(unsigned char key, int x, int y)
{
	switch (key) {
	case 27: /*  Escape key  */
	case 'q': case 'Q': 
		oglExit();
		break;
	default:
		myCtl->handleEvent((int)key);
		break;
	}
}

void oglSetup(const oglOptions &init) {
	int argc=1;
	char *argv[1]={"foo"};
	glutInit(&argc, argv);
	myCfg=init;

/* Create main window */
	int mode=GLUT_DOUBLE | GLUT_RGBA;
	if (myCfg.stencil) mode|=GLUT_STENCIL;
	if (myCfg.depth) mode|=GLUT_DEPTH;
	glutInitDisplayMode (mode);
	glutInitWindowSize (init.width,init.height);
	bool createWindow=true;
	if (myCfg.fullscreen) {
		char modeStr[100];
		sprintf(modeStr,"%dx%d:%d",myCfg.width,myCfg.height,32);
		glutGameModeString(modeStr);
		if (glutEnterGameMode()!=0) /* Game mode worked-- don't open window */
			createWindow=false;
	} 
	if (createWindow) {
		glutCreateWindow (myCfg.name);
	}

	oglCheck("after init");
}

/*  Main Loop
 *  Open window with initial window size, title bar, 
 *  RGBA display mode, and handle input events.
 */
int oglEventloop(oglController *ctl,oglDrawable *scene)
{
	myCtl=ctl;
	myScene=scene;

	myCtl->reshape(myCfg.width,myCfg.height);
	glutMouseFunc(mouseFn);
	glutMotionFunc(motionFn);
	glutReshapeFunc (reshape);
	glutKeyboardFunc (keyboard);
	glutDisplayFunc (display);
	glutMainLoop();
	return 0;
}

void oglExit(void) {
	/* delete myCtl; // Breaks if myScene and myCtl are same object... */
	delete myScene;
	myCtl=0;
	myScene=0;
	exit(0);
}
