/*
Templated radix sort: a fast stable sort, usable with data
that can be decomposed into a small number of keys.

Orion Sky Lawlor, olawlor@acm.org, 2009-03-06 (Public Domain)
*/
#ifndef __OSL_RADIX_SORT_H
#define __OSL_RADIX_SORT_H

#include <string.h> /* for memcpy */

/**
 Templated multi-pass radix sort.
 Sorts the len values in src into dest based
 on the radix key.  Radix keys are generated by 
 the radixKeygen object you pass in.
 
 Returns the number of entries with key KEYS-1.  This can be used
 to delete objects from the array during sorting, by setting their 
 key to KEYS-1.
 
 radixKeygen must have:
   An integer KEYS giving the number of keys, typically around a thousand.
   An integer PASSES giving the number of radix passes (up to 8).
   A method "key(sortType val,unsigned int pass)" that takes a sortType and pass 
     and returns an integer less than KEYS.
 
 Elements will be sorted such that if (key(a,i)<key(b,i)) then 
 a will preceed b in the output.  If the keys are equal, the test
 applies to pass i-1.  That is, pass 0 should return the lowest-order
 bits to compare, pass 1 higher order, and so on (passes are little-endian).
 
 Radix sort is stable.  src is used for temporary space if PASSES>1.
*/
template<class sortType, class radixKeygen>
int radixSort(sortType *src,sortType *dest,unsigned int len,radixKeygen keygen)
{
	/* derived from version by Michael Herf: stereopsis.com */
	unsigned int p,i,k;

	// Histogram maps from a radix key (from radixKey routines)
	//   to a count of the number of elements with that key.
	typedef unsigned int histType;
	const unsigned int KEYS = radixKeygen::KEYS;
	const unsigned int PASSES = radixKeygen::PASSES;
	histType h[PASSES][KEYS];

	// Zero out histograms
	for (p=0;p<PASSES;p++)
	for (k=0;k<KEYS;k++) {
		h[p][k]=(histType)0;
	}
	
	// Build all histograms in one pass
	for (i = 0; i < len; i++)
	for (p=0;p<PASSES;p++) {
		h[p][keygen.key(src[i],p)]++;
	}
	
	// Sum up histograms, and copy over array data 
	int ret=h[PASSES-1][KEYS-1];
	for (p=0;p<PASSES;p++) {
		int sum = 0; /* stores sum of h[0..k-1] */ 
		/* convert h[p][k] from a per-key count to a per-key index */
		for (k = 0; k < KEYS; k++) 
		{ 
			int tsum = h[p][k] + sum; 
			h[p][k] = sum-1; /* the -1 is so we can preincrement below (faster) */ 
			sum = tsum; 
		} 
		sortType *passSrc=src, *passDest=dest;  /* even passes: src -> dest */
		if (p%2==1) {passSrc=dest; passDest=src; } /* odd passes: dest -> src */
		/* Write out elements in sorted order */ 
		for (i = 0; i < len; i++) { 
			sortType s=passSrc[i]; 
			passDest[++(h[p][keygen.key(s,p)])] = s; 
		} 
	}

	if (PASSES%2==0) { // Even number of passes: ended with data in src
		memcpy(dest,src,len*sizeof(sortType));
	}
	
	return ret;
}

/* Generates radix sort keys for a small integer-like object (with bit shift and mask) */
template <class T,int BITS_PER_PASS=8>
class intRadixKeygen {
public:
	/* number of radix passes: rounded-up size of T */
	enum {PASSES=(8*sizeof(T)+BITS_PER_PASS-1)/BITS_PER_PASS};
	enum {KEYS=1<<BITS_PER_PASS};
	unsigned int key(T value,unsigned int pass) {
		/* extract BITS_PER_PASS bits, starting at pass*BITS_PER_PASS: */
		return (value>>(pass*BITS_PER_PASS))&(KEYS-1);
	}
};

#ifdef STANDALONE_TEST
#include <stdio.h>
#include <stdlib.h> /* for rand */
#include <algorithm>
int main(void) {
	enum {n=1000000};
	int a[n], b[n], errcount=0;
	for (int i=0;i<n;i++) a[i]=rand();
	printf("Integer sort takes %d passes...\n",intRadixKeygen<int>::PASSES);
	//std::sort(&a[0],&a[n]); memcpy(b,a,n*sizeof(a[0]));  // reference (2.5x slower)
	radixSort(a,b,n,intRadixKeygen<int>());
	for (int i=1;i<n;i++) {
		if (b[i]<b[i-1]) {
			if (errcount<10) printf("radix sort error! %d after %d\n",b[i],b[i-1]);
			errcount++;
		}
	}
	printf("all %d ints sorted; %d errors\n",n,errcount);
	return 0;
}
#endif

#endif
