/**
  Ambient Occlusion shader
  
  Dr. Orion Sky Lawlor, olawlor@acm.org, 2012-04-11 (Public Domain)
*/
#include "physics/world.h" /* physics::library and physics::object */
#include "ogl/glsl.h"
#include <fstream>
#include <sstream>

/* Just include library bodies here, for easy linking */
#include "physics/world.cpp" 
#include "physics/config.cpp"
#include "ogl/glsl.cpp"

/* Current camera position, for use in raytracers */
vec3 rayObjectCamera;
vec3 lightLocation;
/* Current 4x4 depth projection matrix */
float depthMat[16];

// Check GL for errors after the last command
void oglCheck(const char *where) {
	GLenum e=glGetError();
	if (e==GL_NO_ERROR) return;
	const GLubyte *errString = gluErrorString(e);
	printf("Error %s at %s\n",where, (const char *)errString);
	abort();
}

float randFloat(void) {
	return (rand()&0xffff)*(1.0/(float)0xffff);
}
vec3 randVec(void) {
	return vec3(randFloat(),randFloat(),randFloat());
}


/******************** 3D Mesh Object **********************/
class tri {
public:
	int verts[3];
};

class meshObject : public physics::object {
	std::vector<vec3> verts; // 3D positions
	std::vector<tri> tris; // vertex numbers
	
	std::vector<vec3> movedVerts;
	std::vector<vec3> normals;
public:
	meshObject(const char *objFileName) 
		:physics::object(0.01)  /* <- our timestep, in seconds */
	{
		std::ifstream file("cat.obj");
		std::string line;
		while (std::getline(file,line)) {
			if (line[0]=='v') { /* vertex */
				vec3 v;
				std::stringstream ss(line);
				char c; ss>>c; // skip the 'c'
				ss>>v.x>>v.y>>v.z;

				/* OBJ format has Y up, we want Z up */
				std::swap(v.y,v.z);
				v.y=-v.y;

				/* push model up vertically */
				v.z+=0.2;
				if (!ss.fail())
					verts.push_back(v);
				else {
					std::cout<<"Error in .obj v line!  What is "<<line<<"!?\n";
				}
			}
			else if (line[0]=='f') { /* face: FIXME assumes triangle */
				tri t;
				std::stringstream ss(line);
				char c; ss>>c; // skip the 'f'
				for (unsigned int j=0;j<3;j++)
				{
					ss>>t.verts[j];
					t.verts[j]--; // from 1-based obj file to 0-based truth!
				}
				if (!ss.fail())
					tris.push_back(t);
				else {
					std::cout<<"Error in .obj f line!  What is "<<line<<"!?\n";
				}
			}
		}
		movedVerts.resize(verts.size());
		normals.resize(verts.size());
	}
	
	void simulate(physics::library &lib) { }
	
	void draw(physics::library &lib) {
		
		// Move the vertices
		for (unsigned int i=0;i<verts.size();i++)
			movedVerts[i]=verts[i]+ // old point
					0.1*sin(lib.time)* // how far
						vec3(
							sin(4.0*verts[i].y), // x shift
							0.0, 
							cos(5.0*verts[i].x)); // z shift
		
		// Calculate new normals 
		for (unsigned int i=0;i<verts.size();i++)
			normals[i]=vec3(0.0); // zero out the normal accumulator
		
		for (unsigned int tri=0;tri<tris.size();tri++) 
		{
			vec3 v0=movedVerts[tris[tri].verts[0]];
			vec3 v1=movedVerts[tris[tri].verts[1]];
			vec3 v2=movedVerts[tris[tri].verts[2]];
			vec3 triNormal=normalize(cross(v1-v0,v2-v0));
			for (unsigned int j=0;j<3;j++) {
				normals[tris[tri].verts[j]]+=triNormal;
			}
		}


		/* Load up shader */
		static GLhandleARB triprog=MAKE_PROGRAM_OBJECT3(
		/* vertex program */
			void main(void) {
			   myColor = gl_Color;
			   myNormal = normalize(gl_Normal);
			   /* Include everything in onscreen position */
			   gl_Position=gl_ModelViewProjectionMatrix*gl_Vertex;
			   /* Don't project, but do modelview transforms */
			   location=(vec3)(gl_ModelViewMatrix*gl_Vertex);
			}
		,
		/* shared vertex/fragment code */
			varying vec4 myColor;
			varying vec3 location; // world coordinates of our pixel
			varying vec3 myNormal; 

		,
		/* fragment program */
			uniform sampler2D depthTex[1];
			uniform mat4 depthMat; // takes world coords to depth coords (perspective)
			uniform vec3 camera; // world coordinates of camera
			uniform vec3 lightLocation; // world coordinates of light
			
			/* Return 1.0 if this point is fully visible, 0.0 if fully occluded. */
			float estimate_occlusion(vec3 loc,vec3 N) {
				vec4 dloc4=depthMat*vec4(loc,1.0); /* run shadow location through projection matrix */
				vec3 dloc=vec3(dloc4/dloc4.w)*0.5+0.5; /* perspective divide, and [-1,+1] to [0,1] */
				float mydepth=texture2D(depthTex[0],vec2(dloc));
				
				float total_occlusion=1.0;
				
				for (float angle=0.01234;angle<1.0;angle+=1.0/8.0)
				{
					float a=angle*2.0*3.1415926535;
					vec2 dir=vec2(sin(a),cos(a))*(1.0/1024);
					for (float radius=1.0;radius<=128.0;radius*=2.0)
					{
						float hisdepth=texture2D(depthTex[0],vec2(dloc)+radius*dir);
						if (hisdepth<mydepth) /* he's in front of me */
							total_occlusion-=(1.0/(8.0*7.0));
					}
				}
				
				return total_occlusion;
			}
			
			void main(void) {
				vec3 lightDir=normalize(lightLocation-location);
				vec3 N=normalize(myNormal);
				float ambient=1.0;
				float occlusion=estimate_occlusion(location,N);
				float diffuse=dot(lightDir,N);
				if (diffuse<0.0) diffuse=-0.4*diffuse;
				
				gl_FragColor=vec4(vec3(ambient*occlusion),1.0);
			}
		);
		
		glUseProgramObjectARB(triprog);
		int ul=glGetUniformLocationARB(triprog,"depthTex");
		int depthUnits[1]={7};
		glUniform1ivARB(ul,1,depthUnits); /* depth texture on unit 7 */
		ul=glGetUniformLocationARB(triprog,"depthMat");
		glUniformMatrix4fv(ul,1,0,&depthMat[0]);
		ul=glGetUniformLocationARB(triprog,"camera");
		glUniform3fvARB(ul,1,rayObjectCamera);
		ul=glGetUniformLocationARB(triprog,"lightLocation");
		glUniform3fvARB(ul,1,lightLocation);
		
		/* Draw the triangles */
		glColor3f(1.0,1.0,1.0);
		glBegin(GL_TRIANGLES);
		
		/* Draw mesh */
		for (unsigned int i=0;i<tris.size();i++)
		{
			for (unsigned int j=0;j<3;j++)
			{
				int theVertex=tris[i].verts[j];
				vec3 v=movedVerts[theVertex];
				vec3 normal=normals[theVertex];
				
				glNormal3fv(normalize(normal));
				
				glVertex3fv(v);
			}
		}
		
		/* Draw ground */
		float z=-1.0+0.001;
		float sz=30.0;
		glNormal3f(0,0,1);
		glVertex3f(-sz,-sz,z); glVertex3f(+sz,-sz,z); glVertex3f(+sz,+sz,z); 
		glVertex3f(-sz,-sz,z); glVertex3f(-sz,+sz,z); glVertex3f(+sz,+sz,z); 
		
		glEnd();
		glUseProgramObjectARB(0);
	}
};


/* Called to create a new simulation */
void physics_setup(physics::library &lib) {
	lib.world->add(new meshObject("cat.obj"));
	lib.background[0]=lib.background[1]=0.6;
	lib.background[2]=1.0; // light blue
}

/* Called every frame to draw everything */
void physics_draw_frame(physics::library &lib) {
		oglCheck("before drawing");
	/* Point and line smoothing needs usual alpha blending */
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);
	
	
	int w=1024,h=1024;
	static GLuint texDepth=0;
	if (texDepth==0) {
		oglCheck("before texture work");
		glGenTextures(1,&texDepth);
		glBindTexture(GL_TEXTURE_2D,texDepth);
		glTexImage2D(GL_TEXTURE_2D,0,GL_DEPTH_COMPONENT24, 
				w,h, 0, GL_DEPTH_COMPONENT,GL_FLOAT,0);
		glGenerateMipmap(GL_TEXTURE_2D); //<- slow for depth textures?
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR);
	/*
		glTexParameteri(GL_TEXTURE_2D,
			GL_TEXTURE_COMPARE_MODE_ARB,
			GL_COMPARE_R_TO_TEXTURE_ARB); // magic percent-closer-filtering mode (otherwise, you get black!)
		glTexParameteri(GL_TEXTURE_2D, 
			GL_TEXTURE_COMPARE_FUNC, GL_LEQUAL);
	*/
		oglCheck("after texture work");
	}
	
	static GLuint fbo=0;
	if (fbo==0) {
		glGenFramebuffers(1,&fbo);
		oglCheck("before FBO setup");
		glBindFramebuffer(GL_FRAMEBUFFER,fbo);
		glFramebufferTexture2D(GL_FRAMEBUFFER,
			GL_DEPTH_ATTACHMENT,GL_TEXTURE_2D,texDepth,0);
		glBindFramebuffer(GL_FRAMEBUFFER,0);
		oglCheck("after FBO setup");
	}
	
	oglCheck("before FBO frame");
	glBindFramebuffer(GL_FRAMEBUFFER,fbo);
	glViewport(0,0,w,h);
	glClearColor(1,0,0,0);
	glClear(GL_COLOR_BUFFER_BIT + GL_DEPTH_BUFFER_BIT);
	glEnable(GL_DEPTH_TEST);
	oglCheck("after FBO setup");
	
	float lAngle=lib.time*(1.0/5.0)*3.1415;
	float lRadius=5.0;
	// Light source is circulating around origin
	lightLocation=vec3(lRadius*cos(lAngle),lRadius*sin(lAngle),10.0);
	rayObjectCamera=lightLocation;
	
	glGetFloatv(GL_PROJECTION_MATRIX,&depthMat[0]); /* read back matrix */
	glDisable(GL_TEXTURE_2D);
	

	/* Do depth while drawing objects */
	lib.world->draw(lib);


	glMatrixMode(GL_MODELVIEW);

	glClearDepth(1.0);
	glDepthFunc(GL_LESS);
	oglCheck("after FBO draw");
	glBindFramebuffer(GL_FRAMEBUFFER,0); /* back to onscreen */
	glViewport(0,0,glutGet(GLUT_WINDOW_WIDTH),glutGet(GLUT_WINDOW_HEIGHT));

	glActiveTexture(GL_TEXTURE7);
	glBindTexture(GL_TEXTURE_2D,texDepth);
	glEnable(GL_TEXTURE_2D);
	glActiveTexture(GL_TEXTURE0);
	
	rayObjectCamera=camera;
	/* Do depth while drawing objects */
	glEnable(GL_DEPTH_TEST);
	lib.world->draw(lib);
	
	oglCheck("after drawing");
}


