/**
  Draw multiple objects in a single raytracer pass.
  
  Dr. Orion Sky Lawlor, olawlor@acm.org, 2012-01-30 (Public Domain)
*/
#include "physics/world.h" /* physics::library and physics::object */
#include "ogl/glsl.h"
#include "soil/SOIL.h"

/* Just include library bodies here, for easy linking */
#include "physics/world.cpp" 
#include "physics/config.cpp"
#include "ogl/glsl.cpp"

#include "soil/SOIL.c"
#include "soil/stb_image_aug.c"

/* Make random noise 3D texture */
GLuint makeSolidTexture(int sz) {
	GLuint hdl;
	glGenTextures(1,&hdl);
	glBindTexture(GL_TEXTURE_3D,hdl);
	float *texData=new float[sz*sz*sz];
	for (int z=0;z<sz;z++)
	for (int y=0;y<sz;y++)
	for (int x=0;x<sz;x++)
	{
		vec3 f=vec3(
			x*1.0/sz, /* float 0-1 coordinates */
			y*1.0/sz,
			z*1.0/sz
		);
	
	/* Make some theoretical geometery */
		vec3 c1=vec3(0.3,0.5,0.5);
		float r1=length(f-c1);
		
		vec3 c2=vec3(0.7,0.5,0.5);
		float r2=length(f-c2);
		
		float distance=std::min(r1,r2);
	
	/* Convert final distance to byte */
		texData[x+sz*(y+sz*z)]=distance;
	}
	
	glTexImage3D(GL_TEXTURE_3D,0,
		GL_LUMINANCE16,
		sz,sz,sz,
		0,GL_LUMINANCE,GL_FLOAT,texData);
	glTexParameteri(GL_TEXTURE_3D,GL_TEXTURE_MIN_FILTER,GL_LINEAR);
	glTexParameteri(GL_TEXTURE_3D,GL_TEXTURE_MAG_FILTER,GL_LINEAR);
	GLenum wrapmode=GL_CLAMP_TO_EDGE;
	glTexParameteri(GL_TEXTURE_3D,GL_TEXTURE_WRAP_S,wrapmode);
	glTexParameteri(GL_TEXTURE_3D,GL_TEXTURE_WRAP_T,wrapmode);
	glTexParameteri(GL_TEXTURE_3D,GL_TEXTURE_WRAP_R,wrapmode);
	delete[] texData;
	return hdl;
}

/* A raytraced object */
class rayObject : public physics::object {
public:
	rayObject(void) 
		:physics::object(0.01)  /* <- our timestep, in seconds */
	{ }
	
	void simulate(physics::library &lib) { }
	
	void draw(physics::library &lib) {
		static GLuint texHandle=SOIL_load_OGL_texture(
			"cloud.jpg",
			4, /* RGBA */
			0, /* new texture ID */
			SOIL_FLAG_MIPMAPS | SOIL_FLAG_TEXTURE_REPEATS | SOIL_FLAG_INVERT_Y
		);
		
		//camera.z=-1.0+0.0011; /* height in miles above surface */
		//lib.cameraVelocity=1000.0/3600.0; /* miles/hour  /  seconds/hour */
		
		/* Load up shader */
		static programFromFiles prog; 
		prog.load("vertex.txt","fragment.txt");
		glUseProgramObjectARB(prog);
		unsigned int ul=glGetUniformLocationARB(prog,"camera");
		glUniform3fvARB(ul,1,camera);
		
		ul=glGetUniformLocationARB(prog,"time");
		glUniform1fARB(ul,lib.time);
		
		glActiveTexture(GL_TEXTURE1); /* select texture unit 1 */
		static GLuint cloudHandle=makeSolidTexture(128);
		glBindTexture(GL_TEXTURE_3D,cloudHandle); /* bind handle */
		ul=glGetUniformLocationARB(prog,"blobSampler");
		glUniform1iARB(ul,1); /* set sampler to texture unit 1 */

		/* Draw raytracer proxy geometry *HUGE* (to cover everything) */
		glColor4f(1.0f,1.0f,1.0f,0.4f);
		glutSolidSphere(500.0,4,8);
		
		glUseProgramObjectARB(0);
	}
};


/* Called to create a new simulation */
void physics_setup(physics::library &lib) {
	lib.world->add(new rayObject());
	lib.background[0]=lib.background[1]=0.6;
	lib.background[2]=1.0; // light blue
}

/* Called every frame to draw everything */
void physics_draw_frame(physics::library &lib) {
	/* Point and line smoothing needs usual alpha blending */
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);
	
	lib.world->draw(lib);
}


