/**
  Physics simulator demo program.
  
  Dr. Orion Sky Lawlor, olawlor@acm.org, 2011-02-23 (Public Domain)
*/
#include "physics/world.h"
#include "ogl/glsl.h"
#include "soil/SOIL.h"

/* LAME! just include library bodies here, for easy linking */
#include "physics/world.cpp" 
#include "physics/config.cpp"
#include "ogl/glsl.cpp"
#include "soil/SOIL.c"
#include "soil/stb_image_aug.c"

float fract(float x) {
	return x-floor(x);
}

vec3 fract(const vec3 &src) {
	return vec3(fract(src.x),fract(src.y),fract(src.z));
}


/* Random float between 0 and 1 */
float randf(void) {
        float f=rand();
        return f/RAND_MAX;
}

/* Random vector where x y and z vary between 0 and 1 */
vec3 randVec3(void) {
        return vec3(randf(),randf(),randf());
}


/* One simulated particle */
class particle {
public:
	vec3 color; // RGB
	vec3 pos, vel; // in world coordinates-- meters
	vec3 netforce; // in newtons (quarter-pounds)
	float mass; // in kilograms
	
	void start_step(double dt) {
		/*
		 For stability, I'm using "Leapfrog" integration:
			http://en.wikipedia.org/wiki/Leapfrog_integration
		 Basically, we just update position, *then* calculate
		 net force before moving the velocity.
		*/
		pos+=dt*vel;
		
		/* Zero out net force, so springs can add to it */
		netforce=vec3(0.0);
	}
	
	void finish_step(double dt) {
	/* Newtonian motion */
		vec3 acc=netforce/mass; 

		acc.z+=-9.8;  /* gravity */
		
		/* "wind drag": moderate */
		acc+=-3.2*vel;
		vel+=dt*acc;

		if (pos.z<0.0) { /* hit the ground */
			vel.z=0.9*fabs(vel.z);
			vel*=0.4; // ground drag
			pos.z=0.0;
			//printf("g");
		}
	}
};

/* An object that can push or pull on particles. */
class particlePusher {
public:
	virtual void simulate(double dt,std::vector<particle> &p,physics::library &lib) =0;
	virtual void draw(std::vector<particle> &p,physics::library &lib) =0;
};


/* A chain/rope segment */
class chainSegment : public particlePusher {
	int s,e,n; /* vertex numbers (in particle list) */
	float rest_length;
public:
	
	chainSegment(int s_,int e_,int n_,std::vector<particle> &p)
		:s(s_), e(e_), n(n_)
	{
		rest_length=length(p[e].pos-p[s].pos);
	}
	
	
	void simulate(double dt,std::vector<particle> &p,physics::library &lib) {
		vec3 along=p[e].pos-p[s].pos;
		float len=length(along);
		//if (len>rest_length) { /* pull on neighboring nodes */
			float spring_constant=1.0e3; /* N force per m displacement */
			vec3 pull=(len-rest_length)*spring_constant*normalize(along);
			p[s].netforce+=pull;
			p[e].netforce-=pull;
		//}
		
		if (false) {
			// Avoid bending: wants to lie in straight line
			vec3 nalong=p[n].pos-p[e].pos;
			float dotalong=dot(along,nalong)/(length(along)*length(nalong));
			float bendiness=1.0-dotalong;
			float bend_constant=1.0e1; /* N force per cos of bending (try zero!) */
			vec3 bendforce=bendiness*bend_constant*(nalong-along);
			p[s].netforce-=0.5*bendforce;
			p[e].netforce+=bendforce;
			p[n].netforce-=0.5*bendforce;
		}
	}
	
	void draw(std::vector<particle> &p,physics::library &lib) {
		vec3 spos=p[s].pos, epos=p[e].pos;
		
		/* Draw segment, then shadow */
		for (int shadow=0;shadow<=1;shadow++) {
		  if (shadow) {
		  	glDisable(GL_TEXTURE_2D);
			glDepthMask(GL_FALSE);
			glColor4f(0.0f,0.0f,0.0f,0.3f);
			spos.z=epos.z=0.0; /* squish shadows down */
		  } else { /* ordinary chain */
			glColor3fv(p[s].color);
		  }
		vec3 along=epos-spos;
		vec3 y=camera_orient.y;
		  float r=0.03; // chain link radius
		vec3 x=r*normalize(cross(y,along));
		y=r*normalize(cross(along,x));
		  
		  glBegin(GL_QUADS); //<- move outside draw call?
		  if (s%2) {
		 	glVertex3fv(spos+y); glVertex3fv(epos+y);
		 	glVertex3fv(epos-y); glVertex3fv(spos-y);
		  } else {
		 	glVertex3fv(spos+x); glVertex3fv(epos+x);
		  	glVertex3fv(epos-x); glVertex3fv(spos-x);
		  }
		  glEnd();
		}
		glDepthMask(GL_TRUE);
	}
};



/* A sphere-shaped pusher */
class spherePusher : public particlePusher {
public:
	vec3 location;
	float radius;
	vec4 color;
	spherePusher() {
		location=vec3(0.0);
		radius=1.0;
		color=vec4(1,0,0,0.5);
	}
	
	virtual void push(const vec3 &direction,std::vector<particle> &p) { /* ignore push */ }
	
	virtual void draw(std::vector<particle> &p,physics::library &lib)
	{
		int slices=8,stacks=5;
		for (int pass=1;pass<=3;pass++) {
			glPushMatrix();
			if (pass==3) { glScalef(1,1,0); /* squish shadows onto ground */ }
			glTranslatef(VEC3_TO_XYZ(location));
			
			if (pass!=2) glDepthMask(GL_FALSE); /* transparent -> no depth writes */
			if (pass==1) { /* first pass: transparent outline */
				glColor4fv(color);
				glutSolidSphere(radius,slices,stacks);
			} 
			else if (pass==2) { /* second pass: wireframe */
				glColor4f(0,0,0,1.0);
				glLineWidth(1.0);
				glutWireSphere(radius,slices,stacks);
			} 
			else if (pass==3 && location.z>=0.0) { /* final pass: shadow */
				glColor4f(0,0,0,0.5);
				glutWireSphere(radius,slices,stacks);
				glColor4f(0,0,0,0.3);
				glutSolidSphere(radius,slices,stacks);
			}
			glPopMatrix();
			glDepthMask(GL_TRUE); /* back to normal */
		}
	}
};


/* A force-based particle-repelling sphere */
class softSphere : public spherePusher {
public:
	softSphere(vec3 location_,float radius_,vec4 color_=vec4(1,0,0,0.5)) {
		location=location_; radius=radius_; color=color_;
	}
	
	virtual void simulate(double dt,std::vector<particle> &p,physics::library &lib)
	{
		for (unsigned int i=0;i<p.size();i++)
		{
			vec3 fear=-(location-p[i].pos);
			if (length(fear)<1.0) { // close enough to scare me! 
				p[i].netforce+=normalize(fear)*1000.0*p[i].mass;
			}
		}
	}
};

/* A constraint-based rolling ground sphere */
class rollySphere : public spherePusher {
public:
	int i; /* the particle I am connected to */
	rollySphere(int i_,vec4 color_,std::vector<particle> &p) {
		i=i_;
		location=p[i].pos;
		color=color_;
	}
	
	virtual void simulate(double dt,std::vector<particle> &p,physics::library &lib)
	{
		// Particle should follow our position
		p[i].netforce+=1.0e6*(location-p[i].pos)*p[i].mass;
		
		// Weakly, we should follow the particle's position
		//  location+=0.1*(p[i].pos-location);
		
		
		if (p[i].pos.z<radius) { /* below the ground */
			p[i].pos.z=radius;
			if (p[i].netforce.z<0.0) p[i].netforce.z=0.0; /* no downward forces can apply */
		}
	}
	
	virtual void push(const vec3 &direction,std::vector<particle> &p) 
	{ /* transmit push down to connected node */
		location+=direction; 
	}
};

/* A particle system */
class particlesystem : public physics::object {
	std::vector<particle> p;
	std::vector<particlePusher *> pushers;
	
	unsigned int cur_sphere; /* the sphere we're currently controlling with the camera */
	std::vector<spherePusher *> spheres;
	int n; /* grid of n x n nodes */
public:
	particlesystem(void) 
		:physics::object(0.001)  /* <- our timestep, in seconds */
	{
		/* Make the fabric */
		n=70;  /* grid of n x n links */
		float gs=10.0/n;
		for (int y=0;y<n;y++)  // particle x,y lives at p[y*n+x]
		for (int x=0;x<n;x++) {
			particle np;
			np.color=randVec3();
			np.pos=vec3((x-n/2.0)*gs,y*gs,11.0);
			np.vel=vec3(-2.0,-2.0,2.0); // (randVec3()-vec3(0.5));
			float fabric_density=1.0; // kg per square meter
			np.mass=gs*gs*fabric_density;
			p.push_back(np);
			
			if (x-1>=0 && x+1<n && y-1>=0 && y+1<n)
			{
				pushers.push_back(new chainSegment( // along X 
					y*n+x-1,
					y*n+x,
					y*n+x+1,
					p));
				pushers.push_back(new chainSegment( // along Y 
					(y-1)*n+x,
					y*n+x,
					(y+1)*n+x,
					p));
				pushers.push_back(new chainSegment( // diagonal 1 
					(y-1)*n+x-1,
					y*n+x,
					(y+1)*n+x+1,
					p));
				if (0) 
				pushers.push_back(new chainSegment( // diagonal 2
					(y-1)*n+x+1,
					y*n+x,
					(y+1)*n+x-1,
					p));
			}
		}
		
		/* Make the spheres */
		spherePusher *s;
		
		s=new softSphere(vec3(0.0),1.0,vec4(0.0,1.0,0.0,0.4)); // green
		spheres.push_back(s);
	
		s=new rollySphere(n+1,vec4(1.0,0.0,0.0,0.2),p); // red
		spheres.push_back(s);
		
		s=new rollySphere(n+(n-2),vec4(1.0,0.0,1.0,0.2),p); // purple
		spheres.push_back(s);
		
		cur_sphere=0;
	}
	
	void simulate(physics::library &lib) {
		float slowmo=1.0;
		float dt=timestep.dt*slowmo;
		
		/* Total up kinetic energy of particles */
		double total_energy=0.0;
		for (unsigned int i=0;i<p.size();i++) {
			total_energy+=0.5*p[i].mass*dot(p[i].vel,p[i].vel);
		}
		if (total_energy>1.0e5)
			printf("System kinetic energy: %.1f J\n",total_energy);
		
		// Fix the position of the camera-controlled sphere
		vec3 camera_to_sphere=-camera_orient.z*3.0-camera_orient.y*0.4;
		if (lib.key['f']) { /* flip to next sphere */
			cur_sphere++;
			if (cur_sphere>spheres.size()) cur_sphere=0;
			if (cur_sphere<spheres.size())
				camera=spheres[cur_sphere]->location-camera_to_sphere;
			lib.key['f']=false;
		}
		if (cur_sphere<spheres.size())
			spheres[cur_sphere]->location=camera+camera_to_sphere;
		
		// Start timesteps
		for (unsigned int i=0;i<p.size();i++)
		{
			p[i].start_step(dt);
			//if (length(p[i].vel)>100000.0) {std::cout<<"Unstable at t="<<timestep.last_t<<"\n"; exit(1);}
		}
		
		// Loop over pushers, add in net forces to particles
		for (unsigned int i=0;i<pushers.size();i++) {
			pushers[i]->simulate(dt,p,lib);
		}
		
		// Loop over spheres, simulate and resolve collisions
		for (unsigned int i=0;i<spheres.size();i++) 
		{
		  spheres[i]->simulate(dt,p,lib);
		  for (unsigned int j=i+1;j<spheres.size();j++) {
			vec3 along=spheres[i]->location - spheres[j]->location;
			float len=length(along);
			float minlen=spheres[i]->radius+spheres[j]->radius;
			if (len<minlen) 
			{ /* these spheres are intersecting */
				spheres[i]->push( along*(minlen-len),p);
				spheres[j]->push(-along*(minlen-len),p);
			}
		  }
		}
		
		// Finish timestep
		for (unsigned int i=0;i<p.size();i++) {
			p[i].finish_step(dt);
		}
	}
	
	// Lighting utility functions
	const vec3 &pos(int x,int y) const { return p[y*n+x].pos; }
	vec3 color(int x,int y) const { 
		vec3 xaxis=pos(x+1,y)-pos(x-1,y);
		vec3 yaxis=pos(x,y+1)-pos(x,y-1);
		vec3 normal=normalize(cross(xaxis,yaxis));
		
		vec3 toCamera=camera-pos(x,y); /* toward camera */
		if (dot(normal,toCamera)<0.0f) normal=-normal; /* flip to face camera */
		
		//return normalize(normal);  // color by normal: bright but pretty!
		static vec3 lightDirection=normalize(vec3(0.0,-0.4,1.0));
		float white=dot(normal,lightDirection);
		if (white<0.0) white=-0.3*white; /* facing down: a little reflected light */
		return vec3(white+normal.x*0.1,white+fabs(normal.x)*0.05,white+normal.y*0.1);
	}
	
	void draw(physics::library &lib) {
		// Draw the shadow (essentially just 2D version)
		glDepthMask(GL_FALSE); // avoid Z buffer fighting
		glColor4f(0,0,0,0.7);
		glBegin(GL_TRIANGLE_STRIP);
		for (int y=1;y+2<n;y++) {
			glVertex2fv(pos(1,y));
			int x;
			for (x=1;x+1<n;x++) {
				glVertex2fv(pos(x,y));
				glVertex2fv(pos(x,y+1));
			}
			glVertex2fv(pos(x-1,y+1));
		}
		glEnd();
		glDepthMask(GL_TRUE);
		
		// Draw the fabric
		glBegin(GL_TRIANGLE_STRIP);
		for (int y=1;y+2<n;y++) {
			glVertex3fv(pos(1,y));
			int x;
			for (x=1;x+1<n;x++) {
				glColor3fv(color(x,y));
				glVertex3fv(pos(x,y));
				glColor3fv(color(x,y+1));
				glVertex3fv(pos(x,y+1));
			}
			glVertex3fv(pos(x-1,y+1));
		}
		glEnd();
		
		// Draw all spheres
		for (unsigned int i=0;i<spheres.size();i++)
			spheres[i]->draw(p,lib);
		
		
	}
};


/* Called create a new simulation */
void physics_setup(physics::library &lib) {
	if (!lib.key_toggle['f']) {
		//camera=vec3(0.0,-5.0,1.7);
		camera_orient.x=vec3(1,0,0);
		camera_orient.z=normalize(-vec3(0,0.6,0.4));
	}
	lib.world->add(new particlesystem());
	lib.background[0]=lib.background[1]=0.6;
	lib.background[2]=1.0; // light blue
}

/* Called every frame to draw everything */
void physics_draw_frame(physics::library &lib) {
	/* use SOIL ground texture */
	static GLuint texture=SOIL_load_OGL_texture(
		"ground.jpg",0,0,SOIL_FLAG_MIPMAPS|SOIL_FLAG_TEXTURE_REPEATS
	);
	glBindTexture(GL_TEXTURE_2D,texture);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAX_ANISOTROPY_EXT,32);
	/* Draw opaque white ground plane */
	int groundsize=500;
	glEnable(GL_TEXTURE_2D);
	glEnable(GL_DEPTH_TEST);
	glColor3f(1,1,1);
	glPushMatrix();glTranslatef(0,0,-0.01); /* <- shove down, to reduce Z fighting */
	glBegin(GL_QUADS);
	glTexCoord2i(-groundsize,-groundsize); glVertex2i(-groundsize,-groundsize);
	glTexCoord2i(+groundsize,-groundsize); glVertex2i(+groundsize,-groundsize);
	glTexCoord2i(+groundsize,+groundsize); glVertex2i(+groundsize,+groundsize);
	glTexCoord2i(-groundsize,+groundsize); glVertex2i(-groundsize,+groundsize);
	glEnd();
	glPopMatrix();
	glDisable(GL_TEXTURE_2D);

	/* Point and line smoothing needs normal alpha blending */
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);
	
	/* Now draw objects */
	lib.world->draw(lib);
}


