/**
  Raytracing demo application.
  
  Orion Sky Lawlor, olawlor@acm.org, 2010-01-22 (Public Domain)
*/
#include <iostream>
#include <cmath>
#include <GL/glew.h> /* 
OpenGL Extensions Wrangler:  for gl...ARB extentions.  Must call glewInit after glutCreateWindow! */
#include <GL/glut.h> /* OpenGL Utilities Toolkit, for GUI tools */
#include "ogl/glsl.h"
#include "ogl/glsl.cpp"
#include "ogl/minicam.h"
#include "osl/mat4.h"
#include "osl/mat4_inverse.h"

#include <vector>

// A quadric object
struct quadric_object {
	osl::mat4 A; // quadric surface matrix: 0 = x^T A x
	vec3 reflectance; // diffuse color
	float mirror;  // 0=matte; 1=totally specular reflective
	osl::mat4 MVinv; // from world to trim space
	
	float checkerboard; // 0=uniform color; 1=black and white checkers
	float slice; // slices per meter (0 for no slices)
	float specular; // surface has specular highlight?
	float pad1; // pad to an integeral number of vec4's...
};

void upload_modelview(osl::mat4 &A,quadric_object &q) 
{
	// Object matrix
	osl::mat4 mv; // object -> world space
	glGetFloatv(GL_MODELVIEW_MATRIX,&mv[0][0]);
	osl::mat4 B=inverse(mv); // world -> object
	q.MVinv=B;
	osl::mat4 BAB=transpose(B)*A*B; // see rule to modify quadric object
	q.A=BAB; // upload quadric matrix
}

void display(void) 
{
	glEnable(GL_DEPTH_TEST);
	glDepthMask(GL_TRUE); // turn on Z writes
	glEnable(GL_BLEND);
	
	glClearColor(0.4,0.5,0.7,0.0); // background color
	glClear(GL_COLOR_BUFFER_BIT + GL_DEPTH_BUFFER_BIT);
	
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity(); // flush any ancient matrices
	gluPerspective(70.0, // fov
		glutGet(GLUT_WINDOW_WIDTH)/(float)glutGet(GLUT_WINDOW_HEIGHT),
		0.1,
		1000.0); // z clipping planes
	oglCameraLookAt(1.0);
	
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity(); // flush any ancient matrices
	
	/* The proxy sphere should be gently rotating */
	const float rotate_rate=0.01; /* revolutions per second */
	float angle=glutGet(GLUT_ELAPSED_TIME)*0.001*rotate_rate*360;
	
	/* Build the pixel shader */
	static GLhandleARB prog=makeProgramObjectFromFiles(
		"raytrace_vertex.txt","raytrace.txt");
	glUseProgramObjectARB(prog);
	glFastUniform3fv(prog,"C",1,camera);
	
	// Texture data: RGBA width x height
	std::vector<quadric_object> qvec;

	// Ground plane
	glPushMatrix();
	quadric_object q;
	osl::mat4 A(0.0f);
	A[2][3]=1.0; // z term
	A[3][3]=0.0; // constant term
	upload_modelview(A,q);
	q.reflectance=vec3(0.8,0.8,0.8);
	q.mirror=0.0;
	q.checkerboard=1.0;
	q.slice=0.0;
	q.specular=0.0;
	qvec.push_back(q);
	glPopMatrix();
	
	// Upload the uniform variables for each object:
	for (int obj=0;obj<4;obj++) {
		quadric_object q;
		glPushMatrix(); // keeps each object's transform separate
		if (obj==0) { // funky camera sphere
			vec3 c=camera+camera_orient.z;
			glTranslatef(c.x,c.y,c.z);
			float s=0.8;
			glScalef(s,s,s);
		} else
			glTranslatef(0.0,0.0,1.0+(obj-1.0)*2.0); // translated a bit...
		glRotatef(angle+obj*87,1+obj,0,1); // ...and slowly rotating

		// Quadric surface's matrix:
		osl::mat4 A(0.0f);
		A[0][0]=1.0; // x^2 term
		A[1][1]=1.0; // y^2 term
		A[2][2]=1.0; // z^2 term
		A[2][3]=0.0; // z term
		A[3][3]=-1.0; // constant term: r^2
		
		upload_modelview(A,q);
		
		q.reflectance=vec3(1-0.5*obj,0.5*obj,0.0); // red,green
		q.mirror=0.4;  // reflective spheres
		q.specular=2.0; // specular highlight strength
		q.checkerboard=0.0;
		q.slice=2.3; // slices per meter
		if (obj==0) { /* camera sphere */
			q.mirror=0.0; /* not mirrored */
			q.slice=0.0; /* not sliced */
			q.specular=2.0; /* little specular */
		}
		
		glPopMatrix();
		qvec.push_back(q);
	}
	
/* Upload geometry texture, to texture unit 0 */
	glActiveTexture(GL_TEXTURE0);
	if (sizeof(quadric_object)%sizeof(vec4) !=0) {
		std::cout<<"Sorry, your quadric_object needs to be an integeral number of vec4's\n";
		exit(0);
	}
	glTexImage2D(GL_TEXTURE_RECTANGLE_ARB,0,GL_RGBA32F_ARB, // internal format
		sizeof(quadric_object)/sizeof(vec4),qvec.size(),0, // width x height
		GL_RGBA,GL_FLOAT,&qvec[0]);
	glTexParameteri(GL_TEXTURE_RECTANGLE_ARB,GL_TEXTURE_MAG_FILTER,GL_NEAREST);
	glFastUniform1i(prog,"quadriclist",0); // texture unit 0
	
/* Upload checkerboard texture, to texture unit 1 */
	static bool hasChecker=false;
	if (!hasChecker) { /* only need to build and upload one time; it stays set up. */
		hasChecker=true;
		glActiveTexture(GL_TEXTURE1);
		int checkw=128,checkh=128;
		float *checkarr=new float[checkw*checkh];
		for (int y=0;y<checkh;y++) 
		for (int x=0;x<checkw;x++) {
			float cv=0.0;
			if ((x<checkw/2) ^ (y<checkh/2)) cv=1.0;
			checkarr[x+y*checkw]=cv;
		}
		gluBuild2DMipmaps(GL_TEXTURE_2D,GL_RGBA8,
			checkw,checkh,
			GL_LUMINANCE,GL_FLOAT,checkarr);
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR_MIPMAP_LINEAR);
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR);
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAX_ANISOTROPY_EXT,8);
		delete[] checkarr;
		glFastUniform1i(prog,"checkertex",1); // texture unit 1
	}
	
	// Only draw one set of faces, to avoid drawing raytraced geometry twice.
	//   (front side may get clipped if you're inside the object, so draw back)
	glEnable(GL_CULL_FACE); glCullFace(GL_FRONT);
	glBlendFunc(GL_ONE,GL_ONE_MINUS_SRC_ALPHA); // premultiplied alpha
	
	// Draw one *HUGE* sphere; this one proxy geometry covers all our objects.
	//   Keep in mind the far clipping plane in determining the sphere radius
	glutSolidSphere(200.0,8,6); 
	
	glUseProgramObjectARB(0);
	
	glutPostRedisplay(); // continual animation
	glutSwapBuffers();
}

int main(int argc,char *argv[]) 
{
	glutInit(&argc,argv);
	glutInitDisplayMode(GLUT_RGBA + GLUT_DEPTH + GLUT_DOUBLE);
	glutInitWindowSize(800,600);
	glutCreateWindow("Raytraced Reflections");
	
	glewInit();
	
	glutDisplayFunc(display);
	oglCameraInit();
	camera=vec3(0,-3,1);
	
	glutMainLoop();
	return 0;
}
