/**
  Raytracing demo application.
  
  Orion Sky Lawlor, olawlor@acm.org, 2010-01-22 (Public Domain)
*/
#include <iostream>
#include <cmath>
#include <GL/glew.h> /* 
OpenGL Extensions Wrangler:  for gl...ARB extentions.  Must call glewInit after glutCreateWindow! */
#include <GL/glut.h> /* OpenGL Utilities Toolkit, for GUI tools */
#include "ogl/glsl.h"
#include "ogl/glsl.cpp"
#include "ogl/minicam.h"

void display(void) 
{
	glClearColor(0.4,0.5,0.7,0.0); // background color
	glClear(GL_COLOR_BUFFER_BIT + GL_DEPTH_BUFFER_BIT);
	
	glEnable(GL_DEPTH_TEST);
	glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);
	glEnable(GL_BLEND);
	
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity(); // flush any ancient matrices
	gluPerspective(70.0, // fov
		glutGet(GLUT_WINDOW_WIDTH)/(float)glutGet(GLUT_WINDOW_HEIGHT),
		0.1,
		1000.0); // z clipping planes
	oglCameraLookAt(1.0);
	
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity(); // flush any ancient matrices
	
	/* The proxy sphere should be gently rotating */
	const float rotate_rate=0.1; /* revolutions per second */
	float angle=glutGet(GLUT_ELAPSED_TIME)*0.001*rotate_rate*360;
	
	/* Build a pixel shader */
	static GLhandleARB prog=MAKE_PROGRAM_OBJECT(
		// GLSL vertex shader!
		varying vec3 G; // proxy geometry location, world coordinates
		void main(void) {
			vec4 mv=gl_ModelViewMatrix*gl_Vertex; // apply modelview to get to world coords
			G=vec3(mv);
			gl_Position = gl_ProjectionMatrix*mv;
		}
	,
		// GLSL fragment shader
		uniform vec3 C; // camera position, world coordinates
		varying vec3 G; // proxy geometry location, world coordinates
		uniform float k;
		uniform vec3 obj_center; // object's origin
		
		// Ray-object intersection for hyperboloid.  
		//   Ray is C+t*D, return value is t, or -1 if miss.
		float hyp_intersection(vec3 C,vec3 D) {
			C=C-obj_center; /* move the ray so the object's centered on the origin */
			// Quadratic equation terms:
			float a=D.x*D.x+D.y*D.y-D.z*D.z; // t^2 term
			float b=2.0*C.x*D.x+2.0*C.y*D.y-2.0*C.z*D.z;  // t term
			float c=C.x*C.x+C.y*C.y-C.z*C.z-k; // constant term

			// Solve the quadratic equation:
			float det=b*b-4.0*a*c;
			if (det<0.0) return -1.0; /* miss */
			float t;
			
			// Check both roots
			float sign=-1;
			do {
				t=(-b+sign*sqrt(det))/(2.0*a);
				vec3 P=C+t*D;
				if (abs(P.z)>1.0) t=-1.0; /* trim top of cones */
				if (fract(P.x*10.0)>0.7) t=-1.0; /* add X stripes */
				sign+=2.0;
			} while (t<=0.0 && sign<=1.0);
			return t;
		}
		
		void main(void) {
			float alpha=1.0; // opacity of the final pixel
			
			// Camera ray
			vec3 D=normalize(G-C); // points from camera to proxy
			float t=hyp_intersection(C,D);
			vec3 P=C+t*D;
			
			if (t<=0.0) alpha=0.1; // weakly show proxy geometry
			
			// Compute surface normal
			vec3 oP=P-obj_center;
			vec3 gradient=vec3(2.0*oP.x,2.0*oP.y,-2.0*oP.z);
			vec3 N=normalize(gradient);
			if (dot(N,D)>0.0)
				N=-N; /* was facing wrong way... */
			
			// Compute lighting
			vec3 reflectance=vec3(1,0.3,1);
			vec3 L=normalize(vec3(0.4+3.0*k,0,1.3));
			float light=clamp(dot(N,L),0.0,1.0); // diffuse
			
			vec3 H=normalize(0.5*(L-D)); // Blinn's halfway vector
			float spec=step(0.999,dot(H,N)); // "cartoon highlight"
			spec=pow(spec,200.0); // classic phong exponent
			
			// Shadow ray
			if (hyp_intersection(P,L)>0.01) 
			{  /* we're in shadow */
				light*=0.1;
				spec*=0.1;
			}
			
			light += 0.1; /* ambient */
			
			gl_FragColor = vec4(light*reflectance+spec,alpha);
			// Compute depth, by running intersection point through *remaining* matrices:
			vec4 projP=gl_ProjectionMatrix*vec4(P,1); // already world coordinates: modelview *not* needed
			float depth = projP.z/projP.w; // perspective divide
			gl_FragDepth = 0.5+0.5*depth; // glDepthRange scaling
		}
	);
	glUseProgramObjectARB(prog);
	glFastUniform3fv(prog,"C",1,camera);
	
	// Draw an array of weird hyperboloids:
	vec3 center(0.0);
	for (center.x=-2;center.x<=2;center.x+=2.0)
	for (center.y=0;center.y<=2;center.y+=2.0) {
		glPushMatrix();
		glTranslatef(center.x,center.y,center.z); // move proxy
		glRotatef(angle,0,0,1); // slowly rotate the proxy geometry
		glFastUniform3fv(prog,"obj_center",1,center); // move real geom
		glFastUniform1f(prog,"k",0.2*sin(angle/180.0*3.0+center.x+center.y));
		glutSolidSphere(1.7,8,6); // draw proxy geometry
		glPopMatrix();
	}
	
	glUseProgramObjectARB(0);
	
	/* Big sphere (need some geometry to intersect our objects) */
	glEnable(GL_LINE_SMOOTH);
	glLineWidth(0.8);
	glColor3f(0,1,0);
	glutWireSphere(2.0,30,40);
	
	oglCameraAxes();
	
	/* Grid on the ground */
	double res=10.0;
	glLineWidth(0.1);
	glBegin(GL_LINES);
	glColor3f(0,0,0);
	for (float t=-res;t<=res;t+=1.0) {
		float z=0.0;
		glVertex3f(t,-res,z); // -Y
		glVertex3f(t,+res,z); // +Y
		glVertex3f(-res,t,z); // -X
		glVertex3f(+res,t,z); // +X
		
	}
	glEnd();
	
	
	glutPostRedisplay(); // continual animation
	glutSwapBuffers();
}

int main(int argc,char *argv[]) 
{
	glutInit(&argc,argv);
	glutInitDisplayMode(GLUT_RGBA + GLUT_DEPTH + GLUT_DOUBLE);
	glutInitWindowSize(800,600);
	glutCreateWindow("Raytrace with Shadowing");
	
	glewInit();
	
	glutDisplayFunc(display);
	oglCameraInit();
	camera=vec3(0,-3,1);
	
	glutMainLoop();
	return 0;
}
