/**
 Keep track of events like keydowns, joystick buttons, etc.
 
 Orion Sky Lawlor, olawlor@acm.org, 2006/06/21 (Public Domain)
*/
#include "ogl/event.h"

volatile int oglAsyncRepaintRequest=0;

/* Storage for oglToggles extern.
  This is an array indexed by character number that's 1 when the key is down.
*/
toggle_t oglToggles;

enum {max_buttons=32};
enum {max_axes=32};
static volatile bool joy_buttons[max_buttons];
static volatile float joy_axes[max_axes];
static void update_joy(void);

/**
 Return true if this keyboard key is currently pressed.
 "what" should be a human-readable description of what
 happens when the key gets pressed.
*/
bool oglKey(char c,const char *what) {
	int v=(int)(unsigned char)c;
	if (v<0 || v>=256) return 0;
	return oglToggles[v];
}

/**
 Return true if this joystick/controller button is pressed.
 "what" is a human-readable description of what will happen.
*/
bool oglButton(int n,const char *what) {
	update_joy();
	if (n<0 || n>=max_buttons) return 0;
	return joy_buttons[n];
}

/**
 Return the value of this joystick axis.
 Axes are normalized to return values from -1 to +1, with
 the rest position at 0.0.
*/
double oglAxis(int n,const char *what) {
	update_joy();
	if (n<0 || n>=max_axes) return 0;
	return joy_axes[n];
}

#ifdef __linux__
/******************* Linux Event Interface **************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h> /* for O_RDONLY */
#include <linux/joystick.h>
#include <pthread.h>

/* UNIX file descriptor to open joystick device */
static int joy_fd=0;

extern "C" void close_joy(void) {
	if (joy_fd>0) {close(joy_fd); joy_fd=0;}
}

/* Joystick event handling thread.  This is only needed
  to avoid polling because O_ASYNC doesn't work for joystick devices. */
extern "C" void *joystick_thread(void *) {
	if (joy_fd<=0) return 0;
	printf("Joystick thread running, joystick opened as FD %d\n",joy_fd);
	while (1) {
		struct js_event e;
		if (read(joy_fd,&e,sizeof(e))<=0) continue; /* nothing to grab yet */
		//e.type&=~JS_EVENT_INIT; /* special startup messages */
		int n=e.number+1, v=e.value;
		if (n>max_axes) continue; /* Numbered way too high! */
		if (e.type==JS_EVENT_AXIS) {
			switch (n) {
			case 5: n=4; break; /* fixup axis numbers */
			case 6: n=3; break;
			};
			joy_axes[n]=v*(1.0/0x7fFF);
		} else if (e.type==JS_EVENT_BUTTON) {
			joy_buttons[n]=v;
		} else continue; /* some other weird event */
		
		/* Repaint indefinitely by default */
		oglAsyncRepaintRequest=-1;
		if (v==0) { /* end of a button or axis--one last repaint? */
			for (n=0;n<max_buttons;n++)
				if (joy_buttons[n]!=0) goto keepRepeating;
			for (n=0;n<max_axes;n++)
				if (joy_axes[n]!=0) goto keepRepeating;
			/* All buttons are released--just one more repaint. */
			oglAsyncRepaintRequest=+1;
		keepRepeating: ;
		}
		if (0) printf("Grabbed joystick event: type %d, number %d, value %d; async=%d\n",
			e.type,n,v,oglAsyncRepaintRequest);
	}
}


static int open_joy(void) {
	int fd=open("/dev/input/js0",O_RDONLY);
	if (fd<=0) {
		//printf("Error opening joystick...\n");
		return -1;
	}
	pthread_t pt;
	pthread_create(&pt,NULL,joystick_thread,NULL);
	atexit(close_joy);
	return fd;
}

static void update_joy(void) {
	if (joy_fd==0) joy_fd=open_joy();
}

#else
/******************** Default Implementation of Joystick ***************/
static void update_joy(void) {
	/* do nothing.  All buttons and axes at 0.0 */
}
#endif
