/**
Message-Passing Interface (MPI)-calling 
  Graphics Language User-interface Toolkit (GLUT)
  wrapper.

This code intercepts GLUT calls that set up keyboard 
and mouse listeners, and replaces them with broadcasts and
receives.  This makes it quite easy to run a GLUT application
on a parallel system like a powerwall without changing much code,
but still get excellent performance because rendering is local.

The notion is that *all* these GLUT calls are "collective" calls in MPI.
This lets mpiglut in most cases assume that all processors have 
identical GLUT and OpenGL state.  The only place mpiglut gets in the 
way is to translate coordinates into your local window:
       - Before rendering (but not during!)
       - While receiving mouse events
       - While creating, moving, and querying windows

Orion Sky Lawlor, olawlor@acm.org, 2006/09/19 (Public Domain)
*/
#ifndef __UAF_MPIGLUT_H
#define __UAF_MPIGLUT_H 1
#define MPIGLUT_H 1

#include <mpi.h>
#include <GL/freeglut_std.h> /* or <GL/glut.h>, but *must* use my modified freeglut for now */

#define MGAPI /* empty, but possibly used for Windows DLLs later */
#ifndef MGAPIENTRY
#define MGAPIENTRY /* empty, but possibly used for Windows DLLs later */
#endif

#ifdef __cplusplus
extern "C" {
#endif

/************** mpiglut extensions ***************/

#define MPIGLUT_RECT_VIEWPORT 0xd00face0 /* our backend's rendering viewport, measured in window coordinates */
#define MPIGLUT_RECT_GLOBAL 0xd00face1 /* size of the whole global screen */
#define MPIGLUT_RECT_LOCAL  0xd00face2 /* our backend's display viewport, measured in whole-screen global coordinates */
#define MPIGLUT_RECT_SUBLOCAL 0xd00face3 /* our backend's display viewport, measured in whole-screen global coordinates */

/** Return the coordinates of one of our rectangles.
     value is one of the MPIGLUT_RECT_* types above.
     ret[0] is the left-hand side X coordinate, in pixels.
     ret[1] is the top Y coordinate (increasing downward).
     ret[2] is the width in pixels.
     ret[3] is the height in pixels.
*/
MGAPI void MGAPIENTRY mpiglutGetRect(int value,int *ret);


/** Return the subwindow-from-global-window matrix for the current window and viewport. 
  This matrix returns clip coordinates in our backend's window, given clip coordinates for the virtual global screen.
  Returns a 16-entry matrix in the usual column-major OpenGL fashion.
*/
MGAPI void MGAPIENTRY mpiglutGetSubwindowMatrixd(double *matrix);
MGAPI void MGAPIENTRY mpiglutGetSubwindowMatrixf(float *matrix);


/** Load the subwindow matrix (above) for the current window and viewport
  *if* the GL_PROJECTION matrix is loaded.
  This routine is automatically called by the mpiglut versions of glLoad*, below.
*/
MGAPI void MGAPIENTRY mpiglutLoadSubwindowMatrix( void );


/************ Wrappers around plain glut calls ************/

/** Calls MPI_Init, sets up mpiglut state, and then glutInit */
MGAPI void MGAPIENTRY mpiglutInit( int* pargc, char** argv );

/** Forward a copy of window creation and manipulation requests to the front end */
MGAPI int MGAPIENTRY mpiglutCreateWindow( const char* title );
MGAPI int MGAPIENTRY mpiglutCreateSubWindow( int window, int x, int y, int width, int height );
MGAPI void MGAPIENTRY mpiglutDestroyWindow( int window );
MGAPI void MGAPIENTRY mpiglutSetWindowTitle( const char* title );
MGAPI void MGAPIENTRY mpiglutSetIconTitle( const char* title );
MGAPI void MGAPIENTRY mpiglutReshapeWindow( int width, int height );
MGAPI void MGAPIENTRY mpiglutPositionWindow( int x, int y );
MGAPI void MGAPIENTRY mpiglutShowWindow( void );
MGAPI void MGAPIENTRY mpiglutHideWindow( void );
MGAPI void MGAPIENTRY mpiglutIconifyWindow( void );

/** In GL_PROJECTION mode, these versions  first call glLoadWindowMatrix */
MGAPI void MGAPIENTRY mpiglLoadIdentity( void );
MGAPI void MGAPIENTRY mpiglLoadMatrixd( const GLdouble *m );
MGAPI void MGAPIENTRY mpiglLoadMatrixf( const GLfloat *m );
MGAPI void MGAPIENTRY mpiglLoadTransposeMatrixd( const GLdouble m[16] );
MGAPI void MGAPIENTRY mpiglLoadTransposeMatrixf( const GLfloat m[16] );

/** On the master, broadcast redisplay request to all screens */
MGAPI void MGAPIENTRY mpiglutPostWindowRedisplay( int window );
/** On the master, broadcast redisplay request to all screens */
MGAPI void MGAPIENTRY mpiglutPostRedisplay( void );
/** Optionally synchronize processors before swapping buffers */
MGAPI void MGAPIENTRY mpiglutSwapBuffers( void );

/** Return shift, ctrl, alt state broadcast from the master */
MGAPI int MGAPIENTRY mpiglutGetModifiers(void);

/** We secretly ask OpenGL for only our portion of the viewport */
MGAPI void MGAPIENTRY mpiglViewport(GLint x,GLint y,GLsizei width,GLsizei height);

/** We store the user's display function */
MGAPI void MGAPIENTRY mpiglutDisplayFunc( void (* callback)( void ) );
/** Reshape is called with the size of the entire powerwall's combined screen */
MGAPI void MGAPIENTRY mpiglutReshapeFunc( void (* callback)( int w, int h) );
MGAPI void MGAPIENTRY mpiglutIdleFunc( void (* callback)( void ) );

/** Mouse movement is broadcast from the master */
MGAPI void MGAPIENTRY mpiglutMouseFunc( void (* callback)( int button, int state, int x, int y) );
MGAPI void MGAPIENTRY mpiglutMotionFunc( void (* callback)( int x, int y) );
MGAPI void MGAPIENTRY mpiglutPassiveMotionFunc( void (* callback)( int x, int y) );
MGAPI void MGAPIENTRY mpiglutEntryFunc( void (* callback)( int state) );

/** Keypresses are broadcast from the master */
MGAPI void MGAPIENTRY mpiglutKeyboardFunc( void (* callback)( unsigned char key, int x,int y) );
MGAPI void MGAPIENTRY mpiglutKeyboardUpFunc( void (* callback)( unsigned char key, int x, int y) );
MGAPI void MGAPIENTRY mpiglutSpecialFunc( void (* callback)( int key, int x, int y) );
MGAPI void MGAPIENTRY mpiglutSpecialUpFunc( void (* callback)( int key, int x, int y) );

#ifdef __cplusplus
};
#endif


/* FIXME: 
figure out how to treat windows.
 glutCreateSubWindow, glutReshapeWindow, glutPositionWindow -> should translate coordinates for GLUT
 
Might use glutSetCursor(GLUT_CURSOR_NONE) and glutWarpPointer to extract relative mouse motion, and add up mouse position ourselves.

should probably also handle:
 glutIgnoreKeyRepeat -> send to PE zero (currently OK if called everywhere)
 glutCreateMenu -> stash callback, to allow picking on nonzero PEs

*/

#ifndef MPIGLUT_OVERRIDES_GLUT
/* Route user GLUT calls to mpiglut implementations */
#define glutInit mpiglutInit

#define glutPostWindowRedisplay mpiglutPostWindowRedisplay
#define glutPostRedisplay mpiglutPostRedisplay
#define glutSwapBuffers mpiglutSwapBuffers

#define glutCreateWindow mpiglutCreateWindow
#define glutCreateSubWindow mpiglutCreateSubWindow
#define glutDestroyWindow mpiglutDestroyWindow
#define glutSetWindowTitle mpiglutSetWindowTitle
#define glutSetIconTitle mpiglutSetIconTitle
#define glutReshapeWindow mpiglutReshapeWindow
#define glutPositionWindow mpiglutPositionWindow
#define glutShowWindow mpiglutShowWindow
#define glutHideWindow mpiglutHideWindow
#define glutIconifyWindow mpiglutIconifyWindow

#define glutIconifyWindow mpiglutIconifyWindow

/* If you call mpiglutLoadWindowMatrix yourself, these overloads are never needed */
#ifndef MPIGLUT_MANUAL_WINDOWMATRIX
#define glLoadIdentity mpiglLoadIdentity
#define glLoadMatrixd mpiglLoadMatrixd
#define glLoadMatrixf mpiglLoadMatrixf
#define glLoadTransposeMatrixd mpiglLoadTransposeMatrixd
#define glLoadTransposeMatrixf mpiglLoadTransposeMatrixf
#endif

#define glutGetModifiers mpiglutGetModifiers

#define glViewport mpiglViewport

#define glutDisplayFunc mpiglutDisplayFunc
#define glutReshapeFunc mpiglutReshapeFunc
#define glutIdleFunc mpiglutIdleFunc

#define glutMouseFunc mpiglutMouseFunc
#define glutMotionFunc mpiglutMotionFunc
#define glutPassiveMotionFunc mpiglutPassiveMotionFunc
#define glutEntryFunc mpiglutEntryFunc

#define glutKeyboardFunc mpiglutKeyboardFunc
#define glutKeyboardUpFunc mpiglutKeyboardUpFunc
#define glutSpecialFunc mpiglutSpecialFunc
#define glutSpecialUpFunc mpiglutSpecialUpFunc

#endif /* GLUT overrides */

#endif /* this header */
