/**
  Problem 2:
    This program sorts the data stored in the file 
    "test.1m".  It does this by calling the standard
    C library routine "qsort", which expects its data
    to be in memory.  When qsort accesses a new page
    of memory, the "badMem" page fault routine maps
    that page in from the file.  To keep the total
    amount of memory used small, every time we map
    a new page in, we have to unmap an old page, so
    the total number of pages mapped in is never more
    than 8.
    
    The problem is that we've somehow messed up our 
    program, because we're getting way too many 
    pagefaults-- like a million pagefaults!  Fix the 
    program by changing the "badMem" subroutine to 
    get fewer pagefaults.  (You must still unmap old
    pages, though-- the total number of pages mapped in
    can't be more than 8!)
    
    Once fixed, the program should take run with fewer 
    than 100 pagefaults (it should run for less than 
    about a second), and print out "Problem complete!".
     
  Orion Sky Lawlor, olawlor@acm.org, 2005/4/15 (Public Domain)
*/
#include <stdio.h>
#include <stdlib.h>
#include "badmemlib.h"
#include "badmemlib.cpp"
#include "mmapfile.h"
#include "mmapfile.cpp"

/* Number of page faults */
int nFaults=0;

class myBadMemHandler : public BadMemHandler {
/* Page handling basics */
	MMapFile *f;
	char *base; // Starting address of our region.
	int len; // Size of our region, in bytes
	int pageSize; // Size of one page, in bytes
	
	/* Return the page number that contains this address.
		Returns -1 if there is no such page.
	*/
	int pageFmAddr(void *addr) {
		char *caddr=(char *)addr;
		if (caddr<base) return -1;
		if (caddr>=base+len) return -1;
		return (caddr-base)/pageSize;
	}
	
	/// Map in this page, with this access.
	void map(int page,MMapFile::access_t access) {
		f->map(page*pageSize,access,
			base+page*pageSize,pageSize);
	}
	/// Unmap this page.
	void unmap(int page) {
		f->unmap(base+page*pageSize,pageSize);
	}
	
/* Paging and memory usage control */
	int nKeep;  // Number of pages to keep in memory at once
	enum {nKeepMax=128}; // maximum value for nKeep
	int curPage; // the next page to flush (0..nKeep-1)
	int inmem[nKeepMax]; // lists the pages that are currently mapped into memory
	
public:
	myBadMemHandler(const char *fName,char *base_,int len_,int nKeep_) 
		:base(base_), len(len_)
	{
		f=new MMapFile(fName,MMapFile::readWriteShared,
			base,len);
		pageSize=MMapFile::pageSize;
		nKeep=nKeep_;
		if (nKeep>nKeepMax) {printf("Can only keep %d pages in memory at once!\n",nKeepMax);exit(1);}
		curPage=0;
		/* Initially, there are no pages in memory */
		for (int f=0;f<nKeep;f++) inmem[f]=-1;
	}
	
	/**
	  Hit a bad memory access: find and map in the page we need.
	*/
	virtual void badMem(void *atAddress) {
		nFaults++;
		int page=pageFmAddr(atAddress); /* The page where the fault happened */
		printf("Hit bad memory access at address %p, page %d\n",
			atAddress,page);
		if (page==-1) { /* An actual bad access: */
			printf("Bad memory access at %p.\n",atAddress);
			abort();
		}
		/* Make room for the new page, by getting rid of 
		   one (basically randomly selected) old page. */
		if (inmem[curPage]!=-1) {
			unmap(inmem[curPage]);
		}
		/* Map in the new page
		*/
		map(page,MMapFile::readWriteShared);
		inmem[curPage]=page;
		/* SOLUTION: Next time, unmap the *next* page in the 
		  "inmem" array.  Before, we would unmap the page
		  we had just mapped in!
		*/
		curPage++; if (curPage>=nKeep) curPage=0;
	}
};

/* Comparison function used during QuickSort */
int compare_ints(const void *va,const void *vb) {
	unsigned int a=*(unsigned int *)va;
	unsigned int b=*(unsigned int *)vb;
	if (a<b) return -1;
	if (a>b) return +1;
	return 0;
}

int main(int argc,char *argv[]) {
/* Set up our memory region */
	int nKeep=8; /* Keep up to 8 pages in memory at once */
	if (argc>1) nKeep=atoi(argv[1]);
	char *base_ptr=(char *)0x50000000;
	int len=1*1024*1024;
	myBadMemHandler *h=new myBadMemHandler("test.1m",base_ptr,len,nKeep);
	BadMemSetup(h);
	
/* Sort the data in the memory region */
	qsort(base_ptr,len/sizeof(int),sizeof(int),
		compare_ints);
	
	printf("Data range: %08x %08x\n", 
		((int *)base_ptr)[0], ((int *)base_ptr)[len/sizeof(int)-1]);
	
	printf("Sort complete:  %d pagefaults\n",nFaults);
	if (nFaults<100)
		printf("Problem complete!\n");
	return 0;
}
