/**
  Simple wrapper around operating system's
  memory manipulation routines.
  
  Orion Sky Lawlor, olawlor@acm.org, 2005/4/15 (Public Domain)
*/
#ifndef __OSL_MMAPFILE_H
#define __OSL_MMAPFILE_H

/**
  Represents a file that can be mapped into memory.
*/
class MMapFile {
public:
	/** Machine's page size, in bytes.
	  This is the minimum allocation granularity. 
	  On Windows, this is 64KB, though x86 hardware
	  only requires 4KB.
	*/
	enum {pageSize=64*1024};
	// enum {pageSize=4*1024};
	
	/** Data type for offsets in the file. */
	typedef long long fileOffset;
	
	/** Access to the data in the file. */
	typedef enum {
		noAccess=0,
		readOnly=1,
		readWritePrivate=2,
		readWriteShared=3
	} access_t;
	
	/** Open this file, and prepare to perform mappings
	  in the range from startAddress to startAddress+maxLength.
	  An "fName" of NULL means to request anonymous zero'd memory. 
	*/
	MMapFile(const char *fName,access_t access,
		void *startAddress,int maxLength);
	
	/** Close the file. */
	~MMapFile();
	
	/** Map this portion of the file into memory at this 
	    address. "length" is in bytes, and must be a multiple
	    of pageSize.  This overwrites any mappings that were
	    originally present in that range of memory.
	    If you pass NULL for "toAddress", the system will
	      pick an unused address to map to.
	    The mapped address is returned.
	*/
	void *map(fileOffset fileStart,access_t access,
		void *toAddress,int length);
	
	/** Unmap this portion of the file from memory. */
	void unmap(void *fromAddress,int length);
	
	/** Show the current set of mappings onscreen. */
	static void show(void);
private:
	/* For UNIX, the open file descriptor. */
	int fd;
	/* For windows, the open file and map handle. */
	void *fhandle, *mhandle;
};

#endif

