/**
  Simple wrapper around operating system's
  memory manipulation routines.
  
  Orion Sky Lawlor, olawlor@acm.org, 2005/3/23 (Public Domain)
*/
#include "mmapfile.h"

#ifdef _WIN32
/** Windows Version */
#include <windows.h>
#include <string>

#if 0 /* Demo of how to use anonymous memory */
memory_mapping::memory_mapping(void *req_addr,int req_len) 
{
	/* If specifying an address, *must* reserve
	  address range before putting data there. */
	void *loc1=VirtualAlloc(req_addr,req_len,
		MEM_RESERVE,
		PAGE_NOACCESS);
	/* Now actually put memory at our address */
	addr=VirtualAlloc(req_addr,req_len,
		MEM_COMMIT,
		PAGE_READWRITE);
	if (addr==NULL) { printf("Error %d in VirtualAlloc!\n", GetLastError()); exit(1);}
	len=req_len;
}

memory_mapping::~memory_mapping() {
	VirtualFree(addr,len,MEM_RELEASE);
}
#endif
/** Open this file, and prepare to perform mappings. */
MMapFile::MMapFile(const char *fName,access_t access,
	void *startAddress,int maxLength)
{
	if (fName) {
		int flags=GENERIC_READ;
		if (access>readOnly) flags=flags|GENERIC_WRITE;
		int shflags=0; /* Access available to other processes */
		if (access==readWriteShared) shflags=FILE_SHARE_WRITE;
		fhandle=CreateFile(
			fName,			
			flags,  /* R/W Access I get */			
			shflags, /* (shared) R/W Access others can get */			
			0,             /* Security attributes */			
			OPEN_EXISTING, /* open or create */			
			FILE_ATTRIBUTE_NORMAL, /* Flags and attributes */			
			0); /* Template file */
		if (fhandle==NULL) { printf("Error opening file %s!\n",fName); exit(1); }
	} 
	else { /* fname==NULL: want an anonymous mapping */		
		fhandle=INVALID_HANDLE_VALUE;	
	}
	
	mhandle=CreateFileMapping(		
		fhandle, /* File to map */		
		0, /* SecurityAttributes */		
		PAGE_READWRITE, /* Read/write access */		
		0, /* Object size (high 32 bits) */		
		maxLength, /* Object size (low 32 bits) */		
		0); /* Sharing name */	
	if (mhandle==NULL) { printf("Error %d in CreateFileMapping\n", GetLastError()); exit(1);}
}

/** Close the file. */
MMapFile::~MMapFile()
{
	CloseHandle(mhandle);
	if (fhandle) 
		CloseHandle(fhandle);
}

/** Map this portion of the file into memory at this 
    address. "length" is in bytes, and must be a multiple
    of pageSize.  This overwrites any mappings that were
    originally present in that range of memory.
*/
void *MMapFile::map(fileOffset fileStart,access_t access,
	void *toAddress,int length)
{
	int flags=0;
	switch (access) {
	case noAccess: 
		break;
	case readOnly: 
		flags=FILE_MAP_READ; 
		break;
	case readWritePrivate: 
	case readWriteShared: 
		flags=FILE_MAP_WRITE|FILE_MAP_READ; 
		break;
	};
	void *loc=MapViewOfFileEx(	
		mhandle, /* Handle to file mapping */		
		flags, /* Read/write access */		
		fileStart>>32, /* Offset (high 32 bits) */		
		fileStart, /* Offset (low 32 bits) */		
		length, /* Bytes to map: 0 means whole file */		
		toAddress); /* Suggested address */
	if (toAddress!=NULL && 
		loc!=toAddress) {printf("MapViewOfFileEx to address %p failed!\n",toAddress);exit(1);}
	return loc;
}

/** Unmap this portion of the file from memory. */
void MMapFile::unmap(void *fromAddress,int length)
{
	UnmapViewOfFile(fromAddress);
}

/** Show the current set of mappings. */
void MMapFile::show(void) {
	char *addr=(char *)0x00000000;
	do {
		MEMORY_BASIC_INFORMATION m;
		int len=VirtualQuery(addr,&m,sizeof(m));
		if (len==0) return; /* Kernel region */
		std::string accessType="free";
		
		int ap=m.AllocationProtect;
		if (ap&0x22) accessType="read";
		if (ap&0x44) accessType="read/write";
		if (ap&0x88) accessType="copy-on-write";
		if (ap&0xf0) accessType+=" execute";
		
		if (ap&PAGE_GUARD) accessType+=" guard";
		if (ap&PAGE_NOCACHE) accessType+=" nocache";
		if (ap&PAGE_WRITECOMBINE) accessType+=" writecombine";
		
		if (m.Type&MEM_IMAGE) accessType+=" image";
		if (m.Type&MEM_MAPPED) accessType+=" mapped";
		if (m.Type&MEM_PRIVATE) accessType+=" private";
		
		printf("%p - %p %5d KB: %s\n",
			m.BaseAddress,
			((char *)m.BaseAddress)+m.RegionSize,
			m.RegionSize>>10,
			accessType.c_str());
		addr+=m.RegionSize;
	} while (addr!=0);
}

#else /********************* UNIX Version */
#include <stdio.h>     /* for printf */
#include <stdlib.h>     /* for exit */
#include <sys/mman.h>  /* for mmap */
#include <unistd.h>
#include <errno.h>     /* for perror */
#include <sys/types.h> /* for open */
#include <sys/stat.h>
#include <fcntl.h>



/** Open this file, and prepare to perform mappings. */
MMapFile::MMapFile(const char *fName,access_t access,
	void *startAddress,int maxLength)
{
	if (fName==0) fName="/dev/zero"; /* anonymous memory */
	
	fd=open(fName,O_RDWR,0);
	if (fd<0) {
		perror("error during open:");
		printf("Error while opening file %s!\n",fName);
		exit(1);
	}
}

/** Close the file. */
MMapFile::~MMapFile()
{
	close(fd);
	fd=-1;
}

/** Map this portion of the file into memory at this 
    address. "length" is in bytes, and must be a multiple
    of pageSize.  This overwrites any mappings that were
    originally present in that range of memory.
*/
void *MMapFile::map(fileOffset fileStart,access_t access,
	void *toAddress,int length)
{
	int prot=0;
	int flags=MAP_FIXED;
	if (toAddress==NULL) flags=0; /* Don't map fixed-- let OS decide */
	switch (access) {
	case noAccess: 
		flags|=MAP_PRIVATE; 
		break;
	case readOnly: 
		prot|=PROT_READ; 
		flags|=MAP_SHARED; 
		break;
	case readWritePrivate: 
		prot|=PROT_READ|PROT_WRITE; 
		flags|=MAP_PRIVATE; 
		break;
	case readWriteShared: 
		prot|=PROT_READ|PROT_WRITE; 
		flags|=MAP_SHARED; 
		break;
	}
	
	void *addr=mmap(toAddress, /* Address to map to (must be pagesize multiple!) */
		length, /* Data to map (will be rounded up to pagesize) */
		prot,flags, /* Map flags */
		fd, /* File to map */
		fileStart); /* File offset (bytes) */
	if (addr==(void *)MAP_FAILED) {
		perror("error during mmap:");
		printf("Error during mmap at %p of %d bytes at offset %d\n",
			toAddress,length, (int)fileStart);
		exit(1);
	}
	return addr;
}

/** Unmap this portion of the file from memory. */
void MMapFile::unmap(void *fromAddress,int length)
{
	munmap(fromAddress,length);
}


/** Show the current set of mappings. */
void MMapFile::show(void) {
	char cmd[100];
	sprintf(cmd,"cat /proc/%d/maps",(int)getpid());
	system(cmd);
}
#endif

