/**
  A little library to recieve, and respond to,
  segmentation faults--out of bounds, or bad
  memory accesses.
  
  Orion Sky Lawlor, olawlor@acm.org, 2005/04/15 (Public Domain)
*/
#include "badmemlib.h"
#include <stdio.h>   /* For debugging prints */
#include <stdlib.h>
/* Declare destructor for BadMemHandler class. */
BadMemHandler::~BadMemHandler() {}
/**
 This is the currently installed bad memory handler.
*/
static BadMemHandler *curHandler=NULL;

#ifdef _WIN32 /************ Windows Version *************/

#include <windows.h>
/**
 Signal handler, called by the OS when any exception occurs.
*/
LONG WINAPI
badMemSignal(
    struct _EXCEPTION_POINTERS *ExceptionInfo
    )
{
    EXCEPTION_RECORD *er=ExceptionInfo->ExceptionRecord;
    if (er->ExceptionCode==EXCEPTION_ACCESS_VIOLATION) {
       bool isWrite=er->ExceptionInformation[0];
       curHandler->badMem((void *)er->ExceptionInformation[1]);
       return EXCEPTION_CONTINUE_EXECUTION;
    }
    return EXCEPTION_CONTINUE_SEARCH;
}

/**
  Add a new bad memory handler.
  Returns the previous bad memory handler,
  or NULL if there was no previous handler.
  The library will then call h->badMem on any
  subsequent bad memory accesses.
*/
BadMemHandler *BadMemSetup(BadMemHandler *h) {
    SetUnhandledExceptionFilter(badMemSignal);
	
/* Save the user's BadMemHandler (and return the old one!) */	
	BadMemHandler *oldHandler=curHandler;
	curHandler=h;
	return oldHandler;
}


#else /****************** Not windows-- must be UNIX-like. */

#include <signal.h> /* POSIX signal header. */
#include <errno.h>
#ifdef SOLARIS /* needed with at least Solaris 8 */
#include <siginfo.h>
#endif
#ifdef __APPLE__
#include <ucontext.h>
#endif

/**
 Signal handler, called by the OS when a bad memory
 access occurs.
*/
extern "C"
void badMemSignal(int cause, siginfo_t *si, void *uc_v) 
{
	/* First figure out the address of the bad memory: */
#if defined(__APPLE__) && defined(__ppc__) /* MacOS X on PowerPC: 
	faulting address is in data address register (DAR).  See:
	     /usr/include/ppc/ucontext.h
	     /usr/include/mach/ppc/thread_status.h
	or Section 6.4.3, PowerPC Programming Environments */
	ucontext_t *uc=(ucontext_t *)uc_v;
	// bool isWrite=(1<<6)&uc->uc_mcontext->es.dsisr;
	void *atAddress=(void *)uc->uc_mcontext->es.dar;
#else /* Linux and others have the faulting address right in the siginfo_t */
	void *atAddress=si->si_addr;
#endif

	if (curHandler) { /* Let the user handle this access */
		curHandler->badMem(atAddress);
	} else { /* No handler installed. */
        	printf( "SIGSEGV raised at address %p\n", atAddress);
		exit(1);
	}
}


/**
  Add a new bad memory handler.
  Returns the previous bad memory handler,
  or NULL if there was no previous handler.
  The library will then call h->badMem on any
  subsequent bad memory accesses.
*/
BadMemHandler *BadMemSetup(BadMemHandler *h) {

/* Install our signal handler for SIGSEGV */
        struct sigaction sa;

        sa.sa_sigaction = badMemSignal;
        sigemptyset( &sa.sa_mask );
        sa.sa_flags = 0;
	sa.sa_flags |= SA_SIGINFO; /* we want a siginfo_t */
	sa.sa_flags |= SA_RESTART; /* restart interrupted syscalls (no EINTR errors) */
        if (sigaction (SIGSEGV, &sa, 0)) { perror("sigaction"); exit(1);}
#ifdef SIGBUS
        if (sigaction (SIGBUS, &sa, 0)) { perror("sigaction"); exit(1);}
#endif

/* Save the user's BadMemHandler (and return the old one!) */	
	BadMemHandler *oldHandler=curHandler;
	curHandler=h;
	return oldHandler;
}

#endif

