/**************************************************************************
 *
 * SKT - simple TCP and UDP socket routines that work on 
 * both UNIX (using BSD sockets) and Windows (using WinSock).  
 *
 * Written by Orion Lawlor for the Charm++ Parallel Programming System.
 * Released to public domain 2005/2/9
 *
 *  All port numbers are taken and returned 
 *  in *host* byte order.  This means you can hardcode port
 *  numbers in the code normally, and they will be properly 
 *  translated even on little-endian machines.
 *
 *  SOCKET is just a #define for "unsigned int".
 *
 *  skt_ip_t is a flat bytes structure to hold an IP address--
 *  this is either 4 bytes (for IPv4) or 16 bytes (for IPv6).
 *  It is always in network byte order.
 *
 *  Errors are handled in the library by calling a user-overridable
 *  abort function.
 * 
 * skt_ip_t skt_my_ip(void)
 *   - return the IP address of the current machine.
 *
 * skt_ip_t skt_lookup_ip(const char *name)
 *   - return the IP address of the given machine (DNS or dotted decimal).
 *     Returns 0 on failure.
 *
 * char *skt_print_ip(char *dest,skt_ip_t addr)
 *   - Print the given IP address to the given destination as
 *     dotted decimal.  Dest must be at least 130 bytes long, 
 *     and will be returned.
 *
 * int skt_ip_match(skt_ip_t a,skt_ip_t b)
 *   - Return 1 if the given IP addresses are identical.
 *
 * SOCKET skt_datagram(unsigned int *port, int bufsize)
 *
 *   - creates a UDP datagram socket on the given port.  
 *     Performs the whole socket/bind/getsockname procedure.  
 *     Returns the actual port of the socket and
 *     the file descriptor.  Bufsize, if nonzero, controls the amount
 *     of buffer space the kernel sets aside for the socket.
 *
 * SOCKET skt_server(unsigned int *port)
 *
 *   - create a TCP server socket on the given port (0 for any port).  
 *     Performs the whole socket/bind/listen procedure.  
 *     Returns the actual port of the socket and the file descriptor.
 *
 * SOCKET skt_server_ip(unsigned int *port,skt_ip_t *ip)
 *
 *   - create a TCP server socket on the given port and IP
 *     Use 0 for any port and _skt_invalid_ip for any IP.  
 *     Performs the whole socket/bind/listen procedure.  
 *     Returns the actual port and IP address of the socket 
 *     and the file descriptor.
 *
 * SOCKET skt_accept(SOCKET src_fd,skt_ip_t *pip, unsigned int *port)
 *
 *   - accepts a TCP connection to the specified server socket.  Returns the
 *     IP of the caller, the port number of the caller, and the file
 *     descriptor to talk to the caller.
 *
 * SOCKET skt_connect(skt_ip_t ip, int port, int timeout)
 *
 *   - Opens a TCP connection to the specified server.  Returns a socket for
 *     communication.
 *
 * void skt_close(SOCKET fd)
 *   - Finishes communication on and closes the given socket.
 *
 * int skt_select1(SOCKET fd,int msec)
 *   - Call select on the given socket, returning as soon as
 *     the socket can recv or accept, or (failing that) in the given
 *     number of milliseconds.  Returns 0 on timeout; 1 on readable.
 *
 * int skt_recvN(SOCKET fd,      void *buf,int nBytes)
 * int skt_sendN(SOCKET fd,const void *buf,int nBytes)
 *   - Blocking send/recv nBytes on the given socket.
 *     Retries if possible (e.g., if interrupted), but aborts 
 *     on serious errors.  Returns zero or an abort code.
 *
 * int skt_sendV(SOCKET fd,int nBuffers,void **buffers,int *lengths)
 *   - Blocking call to write from several buffers.  This is much more
 *     performance-critical than read-from-several buffers, because 
 *     individual sends go out as separate network packets, and include
 *     a (35 ms!) timeout for subsequent short messages.  Don't use more
 *     than 8 buffers.
 * 
 * void skt_set_idle(idleFunc f)
 *   - Specify a routine to be called while waiting for the network.
 *     Replaces any previous routine.
 * 
 * void skt_set_abort(abortFunc f)
 *   - Specify a routine to be called when an unrecoverable
 *     (i.e., non-transient) socket error is encountered.
 *     The default is to log the message to stderr and call exit(1).
 *
 **************************************************************************/
#ifndef __SOCK_ROUTINES_H
#define __SOCK_ROUTINES_H

/*Preliminaries*/
#if defined(_WIN32) && ! defined(__CYGWIN__)

  /*For windows systems:*/
#include <winsock.h>
static void sleep(int secs) {Sleep(1000*secs);}

#else
  /*For non-windows (UNIX) systems:*/
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <unistd.h>
#include <fcntl.h>

#ifndef SOCKET
#  define SOCKET int
#  define INVALID_SOCKET (SOCKET)(~0)
#  define SOCKET_ERROR (-1)
#endif /*def SOCKET*/

#endif /*WIN32*/

#ifdef __cplusplus
extern "C" {
#endif

/**
Initialization routine.  You must call this exactly
once at startup on Windows.  Exists, but is a no-op
for UNIX.
*/
void skt_init(void);

/**Error and idle handling*/
typedef void (*skt_idleFn)(void);
void skt_set_idle(skt_idleFn f);

typedef int (*skt_abortFn)(int errCode,const char *msg);
/**
  The abort function is called if something goes wrong in
the socket library.  The default abort function prints out
an error message and exits.
*/
skt_abortFn skt_set_abort(skt_abortFn f);

/**
  This is a network-byte-order IPv4 
  IP address*/
typedef struct { 
	unsigned char data[4];
} skt_ip_t;

/** This is the IP address returned when a lookup fails. */
extern skt_ip_t _skt_invalid_ip;

/** Return the IP address of your own machine.  */
skt_ip_t skt_my_ip(void);
/** Turn this DNS name into an IP address */
skt_ip_t skt_lookup_ip(const char *name);

/** Print this IP address to this string. */
char *skt_print_ip(char *dest,skt_ip_t addr);
/** Return 1 if these are the same IP address */
int skt_ip_match(skt_ip_t a,skt_ip_t b);

/** Build a BSD sockaddr_in from this IP address and port. */
struct sockaddr_in skt_build_addr(skt_ip_t IP,int port);

/** Create a UPD datagram port. */
SOCKET skt_datagram(unsigned int *port, int bufsize);

/** Create a TCP server socket listening on this port
at all IP addresses.  If *port is zero on input, picks
an unused port number.  *port returns the port number used.
*/
SOCKET skt_server(unsigned int *port);

/** Create a TCP server socket listening on this port
for this IP address.  Useful for multi-homed machines. */
SOCKET skt_server_ip(unsigned int *port,skt_ip_t *ip);

/** Accept a TCP connection on this server port.  Returns
the IP address and port of the machine that is connecting
to the server */
SOCKET skt_accept(SOCKET src_fd, skt_ip_t *pip, unsigned int *port);

/** Initiate a TCP connection to this server. */
SOCKET skt_connect(skt_ip_t ip, int port, int timeout);

/** Disconnect from this TCP port. */
void skt_close(SOCKET fd);
/** Return 1 if this TCP port receives data within
these many milliseconds. */
int skt_select1(SOCKET fd,int msec);

/** Set the buffer size for this socket. */
void skt_setSockBuf(SOCKET skt, int bufsize);

/** Send this many bytes to this socket. */
int skt_sendN(SOCKET hSocket,const void *pBuff,int nBytes);
int skt_recvN(SOCKET hSocket,      void *pBuff,int nBytes);
int skt_sendV(SOCKET fd,int nBuffers,const void **buffers,int *lengths);

#ifdef __cplusplus
};
#endif

#endif /*SOCK_ROUTINES_H*/


